"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebExporter = exports.SubprocessExporter = exports.ExternalExporter = exports.convertRequests = exports.listFormats = void 0;
const child_process_1 = __importDefault(require("child_process"));
const base64url_1 = __importDefault(require("base64url"));
const parse_1 = require("./parse");
const python_1 = require("./exporters/python");
const curl_1 = require("./exporters/curl");
const javascript_1 = require("./exporters/javascript");
const php_1 = require("./exporters/php");
const ruby_1 = require("./exporters/ruby");
const util_1 = __importDefault(require("util"));
const isBrowser = typeof window !== "undefined";
const execAsync = !isBrowser ? util_1.default.promisify(child_process_1.default.exec) : undefined;
const EXPORTERS = {
    javascript: new javascript_1.JavaScriptExporter(),
    php: new php_1.PHPExporter(),
    python: new python_1.PythonExporter(),
    ruby: new ruby_1.RubyExporter(),
    curl: new curl_1.CurlExporter(),
};
const LANGUAGES = ["JavaScript", "PHP", "Python", "Ruby", "curl"];
/**
 * Return the list of available export formats.
 *
 * @returns An array of strings with the names of the formats that are available
 *   to use in the `convertRequests()` function.
 */
function listFormats() {
    return LANGUAGES;
}
exports.listFormats = listFormats;
/**
 * Convert Elasticsearch requests in Dev Console syntax to other formats.
 *
 * @param source The source Dev Console code, given as a single string. Multiple
 *   requests can be separated with an empty line.
 * @param outputFormat The format to convert to, such as `"python"` or `"javascript"`.
 * @param options Conversion options.
 * @returns When `checkOnly` is set to `true` in `options`, the return value is a
 *   boolean that indicates if the given source code can be converted. A `false`
 *   return indicates that the requested conversion cannot be completed, for
 *   example due to unsupported features in the specified target format. If
 *   `checkOnly` is `false` or not given, the return value is a string with the
 *   converted code.
 */
async function convertRequests(source, outputFormat, options) {
    const requests = await (0, parse_1.parseRequests)(source);
    const exporter = typeof outputFormat == "string"
        ? EXPORTERS[outputFormat.toLowerCase()]
        : outputFormat;
    if (!exporter) {
        throw new Error("Invalid output format");
    }
    /* istanbul ignore next */
    if (options.debug) {
        console.log(JSON.stringify(requests));
    }
    if (options.checkOnly) {
        return await exporter.check(requests);
    }
    return await exporter.convert(requests, options);
}
exports.convertRequests = convertRequests;
/* this helper function creates a copy of the requests array without the
 * schema request properties, so that the payload sent to external exporters
 * aren't huge.
 */
const getSlimRequests = (requests) => {
    return requests.map((req) => {
        const { request, ...others } = req; /* eslint-disable-line */
        return { ...others };
    });
};
/**
 * Base class for remotely hosted language exporters.
 *
 * This class is used to wrap exporters that are hosted externally,
 * for example as WASM modules.
 * @experimental
 */
class ExternalExporter {
    /**
     * Class constructor.
     *
     * `module` must have `check` and `convert` entry point functions. Both
     * functions must accept a JSON string with the input arguments. The
     * response must be a JSON string with the format `{"return": ..., "error": "..."}`.
     * If the `error` attribute is included, it is assumed that the function
     * failed and an error will be raised with the given error message.
     */
    constructor(module) {
        this._check = module.check;
        this._convert = module.convert;
    }
    async check(requests) {
        const response = JSON.parse(this._check(JSON.stringify({ requests: getSlimRequests(requests) })));
        if (response.error) {
            throw new Error(response.error);
        }
        return response.return;
    }
    async convert(requests, options) {
        const response = JSON.parse(this._convert(JSON.stringify({ requests: getSlimRequests(requests), options })));
        if (response.error) {
            throw new Error(response.error);
        }
        return response.return;
    }
}
exports.ExternalExporter = ExternalExporter;
/**
 * Base class for separate executable language exporters.
 *
 * This class is used to wrap exporters that are hosted externally as
 * independent processes.
 * @experimental
 */
class SubprocessExporter {
    /**
     * Class constructor.
     *
     * `baseCmd` is the base command to run to invoke the exporter. The commands
     * will receive two arguments, first the function to invoke ("check" or
     * "convert") and then the JSON payload that is the input to the function.
     * The function must output the response to stdout in JSON format. A response
     * must be a JSON string with the format `{"return": ..., "error": "..."}`.
     * If the `error` attribute is included, it is assumed that the function
     * failed and an error will be raised with the given error message.
     */
    constructor(baseCmd) {
        this.baseCmd = baseCmd;
    }
    async check(requests) {
        const input = base64url_1.default.encode(JSON.stringify({ requests: getSlimRequests(requests) }));
        if (execAsync === undefined) {
            throw new Error("Cannot use exec()");
        }
        const { stdout, stderr } = await execAsync(`${this.baseCmd} check ${input}`);
        if (stdout) {
            const json = JSON.parse(base64url_1.default.decode(stdout));
            if (json.error) {
                throw new Error(json.error);
            }
            return json.return;
        }
        throw new Error(`Could not invoke exporter: ${stderr}`);
    }
    async convert(requests, options) {
        const input = base64url_1.default.encode(JSON.stringify({ requests: getSlimRequests(requests), options }));
        if (execAsync === undefined) {
            throw new Error("Cannot use exec()");
        }
        const { stdout } = await execAsync(`${this.baseCmd} convert ${input}`);
        if (stdout) {
            const json = JSON.parse(base64url_1.default.decode(stdout));
            if (json.error) {
                throw new Error(json.error);
            }
            return json.return;
        }
        throw new Error("Could not invoke exporter");
    }
}
exports.SubprocessExporter = SubprocessExporter;
/**
 * Base class for web hosted language exporters.
 *
 * This class is used to wrap exporters that are hosted externally as
 * web services.
 * @experimental
 */
class WebExporter {
    /**
     * Class constructor.
     *
     * `baseUrl` is the location where the web service is hosted. The required
     * endpoints are `/check` and `/convert`, appended to this URL. The
     * endpoints must accept a JSON string with the input arguments. The
     * response must be a JSON string with the format `{"return": ..., "error": "..."}`.
     * If the `error` attribute is included, it is assumed that the function
     * failed and an error will be raised with the given error message.
     */
    constructor(baseUrl) {
        this.baseUrl = baseUrl;
    }
    async check(requests) {
        const response = await fetch(`${this.baseUrl}/check`, {
            method: "POST",
            body: JSON.stringify({ requests: getSlimRequests(requests) }),
            headers: { "Content-Type": "application/json" },
        });
        if (response.ok) {
            const json = await response.json();
            if (json.error) {
                throw new Error(json.error);
            }
            return json.return;
        }
        throw new Error("Could not make web request");
    }
    async convert(requests, options) {
        const response = await fetch(`${this.baseUrl}/convert`, {
            method: "POST",
            body: JSON.stringify({ requests: getSlimRequests(requests), options }),
            headers: { "Content-Type": "application/json" },
        });
        if (response.ok) {
            const json = await response.json();
            if (json.error) {
                throw new Error(json.error);
            }
            return json.return;
        }
        throw new Error("Could not make web request");
    }
}
exports.WebExporter = WebExporter;
