"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TrustedDeviceValidator = void 0;
var _securitysolutionListConstants = require("@kbn/securitysolution-list-constants");
var _configSchema = require("@kbn/config-schema");
var _securitysolutionUtils = require("@kbn/securitysolution-utils");
var _base_validator = require("./base_validator");
var _errors = require("./errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Error constants following the established pattern
const TRUSTED_DEVICE_EMPTY_VALUE_ERROR = 'Field value cannot be empty';
const TRUSTED_DEVICE_DUPLICATE_FIELD_ERROR = 'Duplicate field entries are not allowed';
const TRUSTED_DEVICE_DUPLICATE_OS_ERROR = 'Duplicate OS entries are not allowed';
const TRUSTED_DEVICE_USERNAME_OS_ERROR = 'Username field is only supported for Windows OS exclusively. Please select Windows OS only or choose a different field.';
const TrustedDeviceFieldSchema = _configSchema.schema.oneOf([_configSchema.schema.literal(_securitysolutionUtils.TrustedDeviceConditionEntryField.DEVICE_ID), _configSchema.schema.literal(_securitysolutionUtils.TrustedDeviceConditionEntryField.DEVICE_TYPE), _configSchema.schema.literal(_securitysolutionUtils.TrustedDeviceConditionEntryField.HOST), _configSchema.schema.literal(_securitysolutionUtils.TrustedDeviceConditionEntryField.MANUFACTURER), _configSchema.schema.literal(_securitysolutionUtils.TrustedDeviceConditionEntryField.MANUFACTURER_ID), _configSchema.schema.literal(_securitysolutionUtils.TrustedDeviceConditionEntryField.PRODUCT_ID), _configSchema.schema.literal(_securitysolutionUtils.TrustedDeviceConditionEntryField.PRODUCT_NAME), _configSchema.schema.literal(_securitysolutionUtils.TrustedDeviceConditionEntryField.USERNAME)]);
const TrustedDeviceEntrySchema = _configSchema.schema.object({
  field: TrustedDeviceFieldSchema,
  type: _configSchema.schema.oneOf([_configSchema.schema.literal('match'), _configSchema.schema.literal('wildcard'), _configSchema.schema.literal('match_any')]),
  operator: _configSchema.schema.literal('included'),
  value: _configSchema.schema.oneOf([_configSchema.schema.string({
    validate: value => value.trim().length > 0 ? undefined : TRUSTED_DEVICE_EMPTY_VALUE_ERROR
  }), _configSchema.schema.arrayOf(_configSchema.schema.string({
    validate: value => value.trim().length > 0 ? undefined : TRUSTED_DEVICE_EMPTY_VALUE_ERROR
  }), {
    minSize: 1
  })])
});
const TrustedDeviceEntriesSchema = _configSchema.schema.arrayOf(TrustedDeviceEntrySchema, {
  minSize: 1,
  validate(entries) {
    const fields = entries.map(entry => entry.field);
    const duplicateFields = fields.filter((field, index) => fields.indexOf(field) !== index);
    return duplicateFields.length > 0 ? `${TRUSTED_DEVICE_DUPLICATE_FIELD_ERROR}: ${duplicateFields.join(', ')}` : undefined;
  }
});
const TrustedDeviceBasicDataSchema = _base_validator.BasicEndpointExceptionDataSchema.extends({
  osTypes: _configSchema.schema.arrayOf(_configSchema.schema.oneOf([_configSchema.schema.literal(_securitysolutionUtils.OperatingSystem.WINDOWS), _configSchema.schema.literal(_securitysolutionUtils.OperatingSystem.MAC)]), {
    minSize: 1,
    maxSize: 2,
    validate: osTypes => {
      const duplicateOs = osTypes.filter((os, index) => osTypes.indexOf(os) !== index);
      if (duplicateOs.length > 0) {
        return `${TRUSTED_DEVICE_DUPLICATE_OS_ERROR}: ${duplicateOs.join(', ')}`;
      }
      return undefined;
    }
  })
});
const TrustedDeviceDataSchema = _configSchema.schema.object({
  entries: TrustedDeviceEntriesSchema
},
// Because we are only validating some fields from the Exception Item, we set `unknowns` to `ignore` here
{
  unknowns: 'ignore'
});
class TrustedDeviceValidator extends _base_validator.BaseValidator {
  static isTrustedDevice(item) {
    return item.listId === _securitysolutionListConstants.ENDPOINT_ARTIFACT_LISTS.trustedDevices.id;
  }
  async validateHasWritePrivilege() {
    return super.validateHasPrivilege('canWriteTrustedDevices');
  }
  async validateHasReadPrivilege() {
    return super.validateHasPrivilege('canReadTrustedDevices');
  }
  async validateTrustedDevicesFeatureEnabled() {
    if (!this.endpointAppContext.experimentalFeatures.trustedDevices) {
      throw new _errors.EndpointArtifactExceptionValidationError('Trusted devices feature is not enabled');
    }
  }
  async validatePreCreateItem(item) {
    await this.validateTrustedDevicesFeatureEnabled();
    await this.validateHasWritePrivilege();
    await this.validateTrustedDeviceData(item);
    await this.validateCanCreateByPolicyArtifacts(item);
    await this.validateByPolicyItem(item);
    await this.validateCanCreateGlobalArtifacts(item);
    await this.validateCreateOwnerSpaceIds(item);
    return item;
  }
  async validatePreUpdateItem(_updatedItem, currentItem) {
    const updatedItem = _updatedItem;
    await this.validateTrustedDevicesFeatureEnabled();
    await this.validateHasWritePrivilege();
    await this.validateTrustedDeviceData(updatedItem);
    try {
      await this.validateCanCreateByPolicyArtifacts(updatedItem);
    } catch (noByPolicyAuthzError) {
      // Not allowed to create/update by policy data. Validate that the effective scope of the item
      // remained unchanged with this update or was set to `global` (only allowed update). If not,
      // then throw the validation error that was catch'ed
      if (this.wasByPolicyEffectScopeChanged(updatedItem, currentItem)) {
        throw noByPolicyAuthzError;
      }
    }
    await this.validateByPolicyItem(updatedItem, currentItem);
    await this.validateUpdateOwnerSpaceIds(_updatedItem, currentItem);
    await this.validateCanUpdateItemInActiveSpace(_updatedItem, currentItem);
    return _updatedItem;
  }
  async validatePreDeleteItem(currentItem) {
    await this.validateTrustedDevicesFeatureEnabled();
    await this.validateHasWritePrivilege();
    await this.validateCanDeleteItemInActiveSpace(currentItem);
  }
  async validatePreGetOneItem(currentItem) {
    await this.validateTrustedDevicesFeatureEnabled();
    await this.validateHasReadPrivilege();
    await this.validateCanReadItemInActiveSpace(currentItem);
  }
  async validatePreMultiListFind() {
    await this.validateTrustedDevicesFeatureEnabled();
    await this.validateHasReadPrivilege();
  }
  async validatePreExport() {
    await this.validateTrustedDevicesFeatureEnabled();
    await this.validateHasReadPrivilege();
  }
  async validatePreSingleListFind() {
    await this.validateTrustedDevicesFeatureEnabled();
    await this.validateHasReadPrivilege();
  }
  async validatePreGetListSummary() {
    await this.validateTrustedDevicesFeatureEnabled();
    await this.validateHasReadPrivilege();
  }
  async validateTrustedDeviceData(item) {
    try {
      TrustedDeviceBasicDataSchema.validate(item);
    } catch (error) {
      throw new _errors.EndpointArtifactExceptionValidationError(error.message);
    }
    try {
      TrustedDeviceDataSchema.validate(item);
    } catch (error) {
      throw new _errors.EndpointArtifactExceptionValidationError(error.message);
    }
    this.validateOsSpecificFields(item);
  }
  validateOsSpecificFields(item) {
    const osTypes = item.osTypes || [];
    const entries = item.entries || [];
    const hasUsernameField = entries.some(entry => entry.field === _securitysolutionUtils.TrustedDeviceConditionEntryField.USERNAME);
    if (hasUsernameField) {
      // USERNAME field is only allowed for Windows OS exclusively
      if (!(0, _securitysolutionUtils.isTrustedDeviceFieldAvailableForOs)(_securitysolutionUtils.TrustedDeviceConditionEntryField.USERNAME, osTypes)) {
        throw new _errors.EndpointArtifactExceptionValidationError(TRUSTED_DEVICE_USERNAME_OS_ERROR);
      }
    }
  }
}
exports.TrustedDeviceValidator = TrustedDeviceValidator;