"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createTelemetryEndpointTaskConfig = createTelemetryEndpointTaskConfig;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _common = require("@kbn/fleet-plugin/common");
var _types = require("../types");
var _helpers = require("../helpers");
var _configuration = require("../configuration");
var _constants = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Endpoint agent uses this Policy ID while it's installing.
 */
const DefaultEndpointPolicyIdToIgnore = '00000000-0000-0000-0000-000000000000';
const usageLabelPrefix = ['security_telemetry', 'endpoint_task'];
function createTelemetryEndpointTaskConfig(maxTelemetryBatch) {
  const taskType = 'security:endpoint-meta-telemetry';
  return {
    type: taskType,
    title: 'Security Solution Telemetry Endpoint Metrics and Info task',
    interval: '24h',
    timeout: '5m',
    version: '1.0.0',
    getLastExecutionTime: _helpers.getPreviousDailyTaskTimestamp,
    runTask: async (taskId, logger, receiver, sender, taskMetricsService, taskExecutionPeriod) => {
      const mdc = {
        task_id: taskId,
        task_execution_period: taskExecutionPeriod
      };
      const log = (0, _helpers.newTelemetryLogger)(logger.get('endpoint'), mdc);
      const trace = taskMetricsService.start(taskType);
      log.debug('Running telemetry task');
      try {
        const processor = new EndpointMetadataProcessor(log, receiver);
        const documents = await processor.process(taskExecutionPeriod);
        const telemetryUsageCounter = sender.getTelemetryUsageCluster();
        telemetryUsageCounter === null || telemetryUsageCounter === void 0 ? void 0 : telemetryUsageCounter.incrementCounter({
          counterName: (0, _helpers.createUsageCounterLabel)(usageLabelPrefix.concat(['payloads', _types.TelemetryChannel.ENDPOINT_META])),
          counterType: _types.TelemetryCounter.NUM_ENDPOINT,
          incrementBy: documents.length
        });
        log.debug('Sending endpoint telemetry', {
          num_docs: documents.length,
          async_sender: _configuration.telemetryConfiguration.use_async_sender
        });

        // STAGE 6 - Send the documents
        if (_configuration.telemetryConfiguration.use_async_sender) {
          sender.sendAsync(_types.TelemetryChannel.ENDPOINT_META, documents);
        } else {
          const batches = (0, _helpers.batchTelemetryRecords)(documents, maxTelemetryBatch);
          for (const batch of batches) {
            await sender.sendOnDemand(_constants.TELEMETRY_CHANNEL_ENDPOINT_META, batch);
          }
        }
        await taskMetricsService.end(trace);
        return documents.length;
      } catch (error) {
        log.warn(`Error running endpoint alert telemetry task`, {
          error
        });
        await taskMetricsService.end(trace, error);
        return 0;
      }
    }
  };
}
class EndpointMetadataProcessor {
  constructor(logger, receiver) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    this.receiver = receiver;
    this.logger = (0, _helpers.newTelemetryLogger)(logger.get('processor'));
  }
  async process(taskExecutionPeriod) {
    const last = taskExecutionPeriod.last;
    const current = taskExecutionPeriod.current;
    if (!last) {
      throw new Error('last execution timestamp is required');
    }

    // STAGE 1 - Fetch Endpoint Agent Metrics
    const endpointMetrics = await this.receiver.fetchEndpointMetricsAbstract(last, current);
    //  If no metrics exist, early (and successfull) exit
    if (endpointMetrics.totalEndpoints === 0) {
      this.logger.debug('no endpoint metrics to report');
      return [];
    }

    /**
     * STAGE 2
     *  - Fetch Fleet Agent Config
     *  - Ignore policy used while installing the endpoint agent.
     *  - Fetch Endpoint Policy Configs
     */
    const policyIdByFleetAgentId = await this.receiver.fetchFleetAgents().then(policies => {
      policies.delete(DefaultEndpointPolicyIdToIgnore);
      return policies;
    }).catch(error => {
      this.logger.warn('Error fetching fleet agents, using an empty value', {
        error
      });
      return new Map();
    });
    const endpointPolicyById = await this.endpointPolicies(policyIdByFleetAgentId.values());

    /**
     * STAGE 3 - Fetch Endpoint Policy Responses
     */
    const policyResponses = await this.receiver.fetchEndpointPolicyResponses(last, current).then(response => {
      if (response.size === 0) {
        this.logger.info('no endpoint policy responses to report');
      }
      return response;
    }).catch(error => {
      this.logger.warn('Error fetching policy responses, using an empty value', {
        error
      });
      return new Map();
    });

    /**
     * STAGE 4 - Fetch Endpoint Agent Metadata
     */
    const endpointMetadata = await this.receiver.fetchEndpointMetadata(last, current).then(response => {
      if (response.size === 0) {
        this.logger.debug('no endpoint metadata to report');
      }
      return response;
    }).catch(error => {
      this.logger.warn('Error fetching endpoint metadata, using an empty value', {
        error
      });
      return new Map();
    });

    /** STAGE 5 - Create the telemetry log records
     *
     * Iterates through the endpoint metrics documents at STAGE 1 and joins them together
     * to form the telemetry log that is sent back to Elastic Security developers to
     * make improvements to the product.
     */
    const clusterData = await this.fetchClusterData();
    const mappingContext = {
      policyIdByFleetAgentId,
      endpointPolicyById,
      policyResponses,
      endpointMetadata,
      taskExecutionPeriod,
      clusterData
    };
    const telemetryPayloads = [];
    try {
      for await (const metrics of this.receiver.fetchEndpointMetricsById(endpointMetrics.endpointMetricIds)) {
        const payloads = metrics.map(endpointMetric => this.mapEndpointMetric(endpointMetric, mappingContext));
        telemetryPayloads.push(...payloads);
      }
    } catch (error) {
      // something happened in the middle of the pagination, log the error
      // and return what we collect so far instead of aborting the
      // whole execution
      this.logger.warn('Error fetching endpoint metrics by id', {
        error
      });
    }
    return telemetryPayloads;
  }
  async fetchClusterData() {
    const [clusterInfoPromise, licenseInfoPromise] = await Promise.allSettled([this.receiver.fetchClusterInfo(), this.receiver.fetchLicenseInfo()]);
    const clusterInfo = (0, _helpers.safeValue)(clusterInfoPromise);
    const licenseInfo = (0, _helpers.safeValue)(licenseInfoPromise);
    return {
      clusterInfo,
      licenseInfo
    };
  }
  async endpointPolicies(policies) {
    const endpointPolicyCache = new Map();
    for (const policyId of policies) {
      if (!endpointPolicyCache.has(policyId)) {
        const agentPolicy = await this.receiver.fetchPolicyConfigs(policyId).catch(e => {
          this.logger.warn(`error fetching policy config due to ${e === null || e === void 0 ? void 0 : e.message}`);
          return null;
        });
        const packagePolicies = agentPolicy === null || agentPolicy === void 0 ? void 0 : agentPolicy.package_policies;
        if (packagePolicies !== undefined && (0, _helpers.isPackagePolicyList)(packagePolicies)) {
          packagePolicies.map(pPolicy => pPolicy).forEach(pPolicy => {
            var _pPolicy$inputs$, _pPolicy$inputs$2;
            if (((_pPolicy$inputs$ = pPolicy.inputs[0]) === null || _pPolicy$inputs$ === void 0 ? void 0 : _pPolicy$inputs$.config) !== undefined && ((_pPolicy$inputs$2 = pPolicy.inputs[0]) === null || _pPolicy$inputs$2 === void 0 ? void 0 : _pPolicy$inputs$2.config) !== null) {
              pPolicy.inputs.forEach(input => {
                if (input.type === _common.FLEET_ENDPOINT_PACKAGE && (input === null || input === void 0 ? void 0 : input.config) !== undefined && policyId !== undefined) {
                  endpointPolicyCache.set(policyId, pPolicy);
                }
              });
            }
          });
        }
      }
    }
    return endpointPolicyCache;
  }
  mapEndpointMetric(endpointMetric, ctx) {
    var _ctx$clusterData$lice, _policyConfig, _policyConfig$package;
    let policyConfig = null;
    let failedPolicy = null;
    let endpointMetadataById = null;
    const fleetAgentId = endpointMetric.elastic.agent.id;
    const endpointAgentId = endpointMetric.agent.id;
    const policyId = ctx.policyIdByFleetAgentId.get(fleetAgentId);
    if (policyId) {
      policyConfig = ctx.endpointPolicyById.get(policyId) || null;
      if (policyConfig) {
        failedPolicy = ctx.policyResponses.get(endpointAgentId);
      }
    }
    if (ctx.endpointMetadata) {
      endpointMetadataById = ctx.endpointMetadata.get(endpointAgentId);
    }
    const {
      cpu,
      memory,
      uptime,
      documents_volume: documentsVolume,
      malicious_behavior_rules: maliciousBehaviorRules,
      system_impact: systemImpact,
      threads,
      event_filter: eventFilter,
      top_process_trees: topProcessTrees
    } = endpointMetric.Endpoint.metrics;
    const endpointPolicyDetail = (0, _helpers.extractEndpointPolicyConfig)(policyConfig);
    if (endpointPolicyDetail) {
      endpointPolicyDetail.value = (0, _helpers.addDefaultAdvancedPolicyConfigSettings)(endpointPolicyDetail.value);
    }
    return {
      '@timestamp': ctx.taskExecutionPeriod.current,
      cluster_uuid: ctx.clusterData.clusterInfo.cluster_uuid,
      cluster_name: ctx.clusterData.clusterInfo.cluster_name,
      license_id: (_ctx$clusterData$lice = ctx.clusterData.licenseInfo) === null || _ctx$clusterData$lice === void 0 ? void 0 : _ctx$clusterData$lice.uid,
      endpoint_id: endpointAgentId,
      endpoint_version: endpointMetric.agent.version,
      endpoint_package_version: ((_policyConfig = policyConfig) === null || _policyConfig === void 0 ? void 0 : (_policyConfig$package = _policyConfig.package) === null || _policyConfig$package === void 0 ? void 0 : _policyConfig$package.version) || null,
      endpoint_metrics: {
        cpu: cpu.endpoint,
        memory: memory.endpoint.private,
        uptime,
        documentsVolume,
        maliciousBehaviorRules,
        systemImpact,
        threads,
        eventFilter,
        topProcessTrees
      },
      endpoint_meta: {
        os: endpointMetric.host.os,
        capabilities: endpointMetadataById !== null && endpointMetadataById !== undefined ? endpointMetadataById.Endpoint.capabilities : []
      },
      policy_config: endpointPolicyDetail !== null ? endpointPolicyDetail : {},
      policy_response: failedPolicy !== null && failedPolicy !== undefined ? {
        agent_policy_status: failedPolicy.event.agent_id_status,
        manifest_version: failedPolicy.Endpoint.policy.applied.artifacts.global.version,
        status: failedPolicy.Endpoint.policy.applied.status,
        actions: failedPolicy.Endpoint.policy.applied.actions.map(action => action.status !== 'success' ? action : null).filter(action => action !== null),
        configuration: failedPolicy.Endpoint.configuration,
        state: failedPolicy.Endpoint.state
      } : {},
      telemetry_meta: {
        metrics_timestamp: endpointMetric['@timestamp']
      }
    };
  }
}