"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.processLookups = void 0;
var _papaparse = _interopRequireDefault(require("papaparse"));
var _promise_pool = require("../../../../../utils/promise_pool");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const processLookups = async (resources, migrationsClient) => {
  const lookupsData = {};
  resources.forEach(resource => {
    if (resource.type === 'lookup' && !lookupsData[resource.name]) {
      try {
        lookupsData[resource.name] = {
          ...resource,
          data: parseContent(resource.content)
        };
      } catch (error) {
        throw new Error(`Invalid content for lookup ${resource.name}: ${error.message}`);
      }
    }
  });
  const lookups = [];
  const result = await (0, _promise_pool.initPromisePool)({
    concurrency: 10,
    items: Object.entries(lookupsData),
    executor: async ([name, {
      data,
      ...resource
    }]) => {
      if (!data) {
        lookups.push({
          ...resource,
          content: ''
        }); // empty content will make lookup be ignored during translation
        return;
      }
      const indexName = await migrationsClient.create(name, data);
      lookups.push({
        ...resource,
        content: indexName
      }); // lookup will be translated using the index name
    }
  });
  const [error] = result.errors;
  if (error) {
    throw new Error(`Failed to process lookups: ${error.error}`);
  }
  return lookups;
};
exports.processLookups = processLookups;
const parseContent = fileContent => {
  const trimmedContent = fileContent.trim();
  if (trimmedContent === '') {
    return null;
  }
  let arrayContent;
  if (trimmedContent.startsWith('[')) {
    arrayContent = parseJSONArray(trimmedContent);
  } else if (trimmedContent.startsWith('{')) {
    arrayContent = parseNDJSON(trimmedContent);
  } else {
    arrayContent = parseCSV(trimmedContent);
  }
  return arrayContent;
};
const parseCSV = fileContent => {
  const config = {
    header: true,
    // If header is false, rows are arrays; otherwise they are objects of data keyed by the field name.
    skipEmptyLines: true
  };
  const {
    data,
    errors
  } = _papaparse.default.parse(fileContent, config);
  if (errors.length > 0) {
    throw new Error('Invalid CSV');
  }
  return data;
};
const parseNDJSON = fileContent => {
  return fileContent.split(/\n(?=\{)/) // split at newline followed by '{'.
  .filter(entry => entry.trim() !== '') // Remove empty entries.
  .map(parseJSON); // Parse each entry as JSON.
};
const parseJSONArray = fileContent => {
  const parsedContent = parseJSON(fileContent);
  if (!Array.isArray(parsedContent)) {
    throw new Error('invalid JSON');
  }
  return parsedContent;
};
const parseJSON = fileContent => {
  try {
    return JSON.parse(fileContent);
  } catch (error) {
    if (error instanceof RangeError) {
      throw new Error('File is too large');
    }
    throw new Error('Invalid JSON');
  }
};