"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createDeletionDetectionService = void 0;
var _sync_markers = require("../sync_markers/sync_markers");
var _stale_users = require("../../stale_users");
var _utils = require("./utils");
var _bulk = require("../../../bulk");
var _utils2 = require("../../utils");
var _queries = require("./queries");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createDeletionDetectionService = (dataClient, soClient) => {
  const esClient = dataClient.deps.clusterClient.asCurrentUser;
  const syncMarkerService = (0, _sync_markers.createSyncMarkersService)(dataClient, soClient);
  const bulkUtilsService = (0, _bulk.createBulkUtilsService)(dataClient);
  const findStaleUsers = (0, _stale_users.findStaleUsersFactory)(dataClient);
  const deletionDetection = async source => {
    const doFullSync = await syncMarkerService.detectNewFullSync(source);
    const failures = [];
    if (!doFullSync) {
      dataClient.log('debug', `No new full sync for source ${source.id}; skipping deletion detection.`);
      return;
    }
    const [completedEventTimeStamp, startedEventTimeStamp] = await Promise.all([syncMarkerService.findLastEventMarkerInIndex(source, 'completed'), syncMarkerService.findLastEventMarkerInIndex(source, 'started')]);
    dataClient.log(`debug`, `Full Sync for source ${source.id}, starting deletion detection.`);
    if ((await (0, _utils.timeStampsAreValid)({
      completedEventTimeStamp,
      startedEventTimeStamp,
      source,
      dataClient
    })) && completedEventTimeStamp !== undefined && startedEventTimeStamp !== undefined) {
      // Proceed with deletion detection
      dataClient.log('debug', `Full sync detected for ${source.id}. Running deletion detection...`);
      // get all users in the integrations docs between the two timestamps
      const allIntegrationsUserNames = await getUsersInFullSync({
        source,
        startedEventTimeStamp,
        completedEventTimeStamp
      });

      // get all users in the privileged index for this source that are not in integrations docs
      const staleUsers = await findStaleUsers(source.id, allIntegrationsUserNames, 'entity_analytics_integration');
      if (staleUsers.length === 0) {
        dataClient.log('debug', `No stale users to soft delete for source ${source.id}`);
        return;
      }
      const ops = bulkUtilsService.bulkSoftDeleteOperations(staleUsers, dataClient.index, 'entity_analytics_integration');
      try {
        // soft delete the users
        const resp = await esClient.bulk({
          body: ops,
          refresh: 'wait_for'
        });
        failures.push(...(0, _utils2.getErrorFromBulkResponse)(resp));
      } catch (error) {
        dataClient.log('error', `Error executing bulk operations: ${error}`);
      }
      if (failures.length > 0) {
        dataClient.log('error', `${failures.length} errors upserting users with bulk operations.
          The first error is: ${JSON.stringify(failures[0])}`);
      }
    }
  };
  const getUsersInFullSync = async ({
    completedEventTimeStamp,
    startedEventTimeStamp,
    source
  }) => {
    let afterKey;
    const pageSize = 100;
    let fetchMore = true;
    const usersToDelete = [];
    while (fetchMore) {
      var _privilegedMonitoring, _privilegedMonitoring2, _privilegedMonitoring3, _privilegedMonitoring4;
      const privilegedMonitoringUsers = await esClient.search({
        index: source.indexPattern,
        ...(0, _queries.buildFindUsersSearchBody)({
          timeGte: startedEventTimeStamp,
          timeLt: completedEventTimeStamp,
          afterKey,
          pageSize
        })
      });
      const buckets = privilegedMonitoringUsers === null || privilegedMonitoringUsers === void 0 ? void 0 : (_privilegedMonitoring = privilegedMonitoringUsers.aggregations) === null || _privilegedMonitoring === void 0 ? void 0 : (_privilegedMonitoring2 = _privilegedMonitoring.users) === null || _privilegedMonitoring2 === void 0 ? void 0 : _privilegedMonitoring2.buckets;
      if (buckets) {
        usersToDelete.push(...buckets.map(bucket => bucket.key.username));
      }
      afterKey = (_privilegedMonitoring3 = privilegedMonitoringUsers.aggregations) === null || _privilegedMonitoring3 === void 0 ? void 0 : (_privilegedMonitoring4 = _privilegedMonitoring3.users) === null || _privilegedMonitoring4 === void 0 ? void 0 : _privilegedMonitoring4.after_key;
      fetchMore = Boolean(afterKey);
    }
    return usersToDelete;
  };
  return {
    deletionDetection
  };
};
exports.createDeletionDetectionService = createDeletionDetectionService;