"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createIndexSyncService = void 0;
var _esQuery = require("@kbn/es-query");
var _lodash = require("lodash");
var _search = require("../../users/search");
var _saved_objects = require("../../saved_objects");
var _bulk = require("../bulk");
var _stale_users = require("./stale_users");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createIndexSyncService = (dataClient, maxUsersAllowed) => {
  const {
    deps
  } = dataClient;
  const esClient = deps.clusterClient.asCurrentUser;
  const bulkUtilsService = (0, _bulk.createBulkUtilsService)(dataClient);
  const findStaleUsers = (0, _stale_users.findStaleUsersFactory)(dataClient);
  const searchService = (0, _search.createSearchService)(dataClient);

  /**
   * Synchronizes users from monitoring index sources and soft-deletes (mark as not privileged) stale entries.
   *
   * This method:
   * - Retrieves all saved objects of type 'index' that define monitoring sources.
   * - For each valid source with an index pattern, fetches usernames from the monitoring index.
   * - Identifies users no longer present in the source index (stale users).
   * - Performs a bulk soft-delete (marks as not privileged) for all stale users found.
   * - Handles missing indices gracefully by logging a warning and skipping them.
   *
   * Additionally, all users from index sources are synced with the internal privileged user index,
   * ensuring each user is either created or updated with the latest data.
   *
   * @returns {Promise<void>} Resolves when synchronization and soft-deletion are complete.
   */
  const plainIndexSync = async soClient => {
    const monitoringIndexSourceClient = new _saved_objects.MonitoringEntitySourceDescriptorClient({
      soClient,
      namespace: deps.namespace
    });
    // get all monitoring index source saved objects of type 'index_sync'
    const indexSources = await monitoringIndexSourceClient.findSourcesByType('index');
    if (indexSources.length === 0) {
      dataClient.log('debug', 'No monitoring index sources found. Skipping sync.');
      return;
    }
    dataClient.log('info', `Privilege monitoring sync started - Maximum supported number of privileged users allowed: ${maxUsersAllowed}`);
    const allStaleUsers = [];
    for (const source of indexSources) {
      // eslint-disable-next-line no-continue
      if (!source.indexPattern) continue; // if no index pattern, skip this source
      const index = source.indexPattern;
      try {
        var _source$filter;
        const allUserNames = await syncUsernamesFromIndex({
          indexName: index,
          sourceId: source.id,
          kuery: (_source$filter = source.filter) === null || _source$filter === void 0 ? void 0 : _source$filter.kuery
        });

        // collect stale users
        const staleUsers = await findStaleUsers(source.id, allUserNames, 'index');
        allStaleUsers.push(...staleUsers);
      } catch (error) {
        var _error$meta, _error$meta$body, _error$meta$body$erro, _error$message;
        if ((error === null || error === void 0 ? void 0 : (_error$meta = error.meta) === null || _error$meta === void 0 ? void 0 : (_error$meta$body = _error$meta.body) === null || _error$meta$body === void 0 ? void 0 : (_error$meta$body$erro = _error$meta$body.error) === null || _error$meta$body$erro === void 0 ? void 0 : _error$meta$body$erro.type) === 'index_not_found_exception' || error !== null && error !== void 0 && (_error$message = error.message) !== null && _error$message !== void 0 && _error$message.includes('index_not_found_exception')) {
          dataClient.log('warn', `Index "${index}" not found — skipping.`);
          // eslint-disable-next-line no-continue
          continue;
        }
        dataClient.log('error', `Unexpected error during sync for index "${index}": ${error.message}`);
      }
    }
    // Soft delete stale users
    dataClient.log('debug', `Found ${allStaleUsers.length} stale users across all index sources.`);
    if (allStaleUsers.length > 0) {
      const ops = bulkUtilsService.bulkSoftDeleteOperations(allStaleUsers, dataClient.index, 'index');
      const resp = await esClient.bulk({
        body: ops,
        refresh: 'wait_for'
      });
      const errors = (0, _utils.getErrorFromBulkResponse)(resp);
      if (errors.length > 0) {
        dataClient.log('error', `${errors.length} errors deleting stale users with bulk operation.
        The first error is: ${JSON.stringify(errors[0])}`);
        // Log all errors for debugging
        dataClient.log('debug', `Errors deleting stale users: ${JSON.stringify(errors)}`);
      }
    }
  };

  /**
   * Synchronizes usernames from a specified index by collecting them in batches
   * and performing create or update operations in the privileged user index.
   *
   * This method:
   * - Executes a paginated search on the provided index (with optional KQL filter).
   * - Extracts `user.name` values from each document.
   * - Checks for existing monitored users to determine if each username should be created or updated.
   * - Performs bulk operations to insert or update users in the internal privileged user index.
   *
   * Designed to support large indices through pagination (`search_after`) and batching.
   * Logs each step and handles errors during bulk writes.
   *
   * @param indexName - Name of the Elasticsearch index to pull usernames from.
   * @param kuery - Optional KQL filter to narrow down results.
   * @returns A list of all usernames processed from the source index.
   */
  const syncUsernamesFromIndex = async ({
    indexName,
    sourceId,
    kuery
  }) => {
    const allUsernames = []; // Collect all usernames across batches
    let searchAfter;
    const batchSize = 100;
    const query = kuery ? (0, _esQuery.toElasticsearchQuery)((0, _esQuery.fromKueryExpression)(kuery)) : {
      match_all: {}
    };
    const failures = [];
    while (true) {
      const response = await searchService.searchUsernamesInIndex({
        indexName,
        batchSize,
        searchAfter,
        query
      });
      const hits = response.hits.hits;
      if (hits.length === 0) break;
      const batchUsernames = hits.map(hit => {
        var _hit$_source, _hit$_source$user;
        return (_hit$_source = hit._source) === null || _hit$_source === void 0 ? void 0 : (_hit$_source$user = _hit$_source.user) === null || _hit$_source$user === void 0 ? void 0 : _hit$_source$user.name;
      }).filter(username => !!username);
      allUsernames.push(...batchUsernames); // Collect usernames from this batch
      const batchUniqueUsernames = (0, _lodash.uniq)(batchUsernames); // Ensure uniqueness within the batch

      dataClient.log('debug', `Found ${batchUniqueUsernames.length} unique usernames in ${batchUsernames.length} hits.`);
      const existingUserMap = await searchService.getExistingUsersMap(batchUsernames);
      const usersToWrite = batchUniqueUsernames.map(username => ({
        username,
        sourceId,
        existingUserId: existingUserMap.get(username),
        isPrivileged: true
      }));
      if (usersToWrite.length === 0) return batchUsernames;
      const ops = bulkUtilsService.bulkUpsertOperations(usersToWrite, dataClient.index);
      dataClient.log('debug', `Executing bulk operations for ${usersToWrite.length} users`);
      try {
        dataClient.log('debug', `Bulk ops preview:\n${JSON.stringify(ops, null, 2)}`);
        const resp = await esClient.bulk({
          body: ops,
          refresh: true
        });
        failures.push(...(0, _utils.getErrorFromBulkResponse)(resp));
      } catch (error) {
        dataClient.log('error', `Error executing bulk operations: ${error}`);
      }
      searchAfter = hits[hits.length - 1].sort;
    }
    if (failures.length > 0) {
      dataClient.log('error', `${failures.length} errors upserting users with bulk operations.
        The first error is: ${JSON.stringify(failures[0])}`);

      // Log all errors for debugging
      dataClient.log('debug', `Errors upserting users: ${JSON.stringify(failures)}`);
    }
    return (0, _lodash.uniq)(allUsernames); // Return all unique usernames collected across batches;
  };
  return {
    plainIndexSync,
    _syncUsernamesFromIndex: syncUsernamesFromIndex
  };
};
exports.createIndexSyncService = createIndexSyncService;