"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initEntityEngineRoute = void 0;
var _utils = require("@kbn/lists-plugin/server/routes/utils");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _zodHelpers = require("@kbn/zod-helpers");
var _privileges = require("../../../../../common/entity_analytics/privileges");
var _search_strategy = require("../../../../../common/search_strategy");
var _init = require("../../../../../common/api/entity_analytics/entity_store/engine/init.gen");
var _constants = require("../../../../../common/constants");
var _check_and_init_asset_criticality_resources = require("../../asset_criticality/check_and_init_asset_criticality_resources");
var _validation = require("./validation");
var _utils2 = require("../utils");
var _check_and_init_privmon_resources = require("../../privilege_monitoring/check_and_init_privmon_resources");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const initEntityEngineRoute = (router, logger, config) => {
  router.versioned.post({
    access: 'public',
    path: '/api/entity_store/engines/{entityType}/init',
    security: {
      authz: {
        requiredPrivileges: ['securitySolution', `${_constants.APP_ID}-entity-analytics`]
      }
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: {
        params: (0, _zodHelpers.buildRouteValidationWithZod)(_init.InitEntityEngineRequestParams),
        body: (0, _validation.buildInitRequestBodyValidation)(_init.InitEntityEngineRequestBody)
      }
    }
  }, async (context, request, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    const secSol = await context.securitySolution;
    const {
      pipelineDebugMode
    } = config.entityAnalytics.entityStore.developer;
    const {
      getSpaceId,
      getAppClient,
      getDataViewsService
    } = await context.securitySolution;
    const entityStoreClient = secSol.getEntityStoreDataClient();
    try {
      const securitySolutionIndices = await (0, _utils2.buildIndexPatternsByEngine)(getSpaceId(), _search_strategy.EntityType[request.params.entityType], getAppClient(), getDataViewsService());
      const privileges = await entityStoreClient.getEntityStoreInitPrivileges(securitySolutionIndices);
      if (!privileges.has_all_required) {
        const missingPrivilegesMsg = (0, _privileges.getMissingPrivilegesErrorMessage)((0, _privileges.getAllMissingPrivileges)(privileges));
        return siemResponse.error({
          statusCode: 403,
          body: `User does not have the required privileges to initialize the entity engine\n${missingPrivilegesMsg}`
        });
      }
      await (0, _check_and_init_asset_criticality_resources.checkAndInitAssetCriticalityResources)(context, logger);
      await (0, _check_and_init_privmon_resources.checkAndInitPrivilegeMonitoringResources)(context, logger);
      const body = await entityStoreClient.init(_search_strategy.EntityType[request.params.entityType], request.body, {
        pipelineDebugMode
      });
      return response.ok({
        body
      });
    } catch (e) {
      const error = (0, _securitysolutionEsUtils.transformError)(e);
      logger.error(`Error initialising entity engine: ${error.message}`);
      return siemResponse.error({
        statusCode: error.statusCode,
        body: error.message
      });
    }
  });
};
exports.initEntityEngineRoute = initEntityEngineRoute;