"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.assetCriticalityServiceFactory = void 0;
var _fp = require("lodash/fp");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isCriticalityIdentifierValid = identifier => !(0, _fp.isEmpty)(identifier.id_field) && !(0, _fp.isEmpty)(identifier.id_value);
const groupIdentifierValuesByField = identifiers => identifiers.reduce((acc, id) => {
  acc[id.id_field] ??= [];
  if (!acc[id.id_field].includes(id.id_value)) {
    acc[id.id_field].push(id.id_value);
  }
  return acc;
}, {});
const buildCriticalitiesQuery = identifierValuesByField => ({
  bool: {
    filter: {
      bool: {
        should: Object.keys(identifierValuesByField).map(idField => ({
          bool: {
            must: [{
              term: {
                id_field: idField
              }
            }, {
              terms: {
                id_value: identifierValuesByField[idField]
              }
            }]
          }
        }))
      }
    }
  }
});
const getCriticalitiesByIdentifiers = async ({
  assetCriticalityDataClient,
  identifiers
}) => {
  if (identifiers.length === 0) {
    throw new Error('At least one identifier must be provided');
  }
  const validIdentifiers = identifiers.filter(id => isCriticalityIdentifierValid(id));
  if (validIdentifiers.length === 0) {
    throw new Error('At least one identifier must contain a valid field and value');
  }
  const identifierCount = validIdentifiers.length;
  const identifierValuesByField = groupIdentifierValuesByField(validIdentifiers);
  const criticalitiesQuery = buildCriticalitiesQuery(identifierValuesByField);
  const criticalitySearchResponse = await assetCriticalityDataClient.search({
    query: criticalitiesQuery,
    size: identifierCount
  });

  // @ts-expect-error @elastic/elasticsearch _source is optional
  return criticalitySearchResponse.hits.hits.map(hit => hit._source);
};
const assetCriticalityServiceFactory = ({
  assetCriticalityDataClient
}) => ({
  getCriticalitiesByIdentifiers: identifiers => getCriticalitiesByIdentifiers({
    assetCriticalityDataClient,
    identifiers
  })
});
exports.assetCriticalityServiceFactory = assetCriticalityServiceFactory;