"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.robustUnset = exports.robustSet = exports.robustGet = void 0;
var _saferLodashSet = require("@kbn/safer-lodash-set");
var _lodash = require("lodash");
var _is_objectlike_or_array_of_objectlikes = require("./is_objectlike_or_array_of_objectlikes");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Similar to lodash `get`, but instead of handling only pure dot or nested notation this function handles any mix of dot and nested notation
 *
 * Note: this function makes no attempt to handle arrays in the middle of the path because it's only used to fetch values based on paths from
 * the `fields` option on search requests, which can never have arrays in the middle of the path
 * @param key Path to field, in dot notation
 * @param document Object to fetch value from
 * @returns Fetched value or undefined
 */
const robustGet = ({
  key,
  document
}) => {
  const fastPathValue = document[key];
  if (fastPathValue != null) {
    return fastPathValue;
  }
  const splitKey = key.split('.');
  let tempKey = splitKey[0];
  for (let i = 0; i < splitKey.length - 1; i++) {
    if (i > 0) {
      tempKey += `.${splitKey[i]}`;
    }
    const value = document[tempKey];
    if (value != null) {
      if ((0, _is_objectlike_or_array_of_objectlikes.isObjectTypeGuard)(value)) {
        return robustGet({
          key: splitKey.slice(i + 1).join('.'),
          document: value
        });
      } else {
        return undefined;
      }
    }
  }
  return undefined;
};

/**
 * Similar to lodash set, but instead of handling only pure dot or nested notation this function handles any mix of dot and nested notation
 * @param key Path to field, in dot notation
 * @param valueToSet Value to insert into document
 * @param document Object to insert value into
 * @returns Updated document
 */
exports.robustGet = robustGet;
const robustSet = ({
  key,
  valueToSet,
  document
}) => {
  const splitKey = key.split('.');
  let tempKey = splitKey[0];
  for (let i = 0; i < splitKey.length - 1; i++) {
    if (i > 0) {
      tempKey += `.${splitKey[i]}`;
    }
    const value = document[tempKey];
    if (value != null) {
      if ((0, _is_objectlike_or_array_of_objectlikes.isObjectTypeGuard)(value)) {
        robustSet({
          key: splitKey.slice(i + 1).join('.'),
          valueToSet,
          document: value
        });
        return document;
      }
    }
  }
  return (0, _saferLodashSet.set)(document, key, valueToSet);
};

/**
 * Similar to lodash unset, but instead of handling only pure dot or nested notation this function handles any mix of dot and nested notation
 * @param key Path to field, in dot notation
 * @param document Object to insert value into
 * @returns updated document
 */
exports.robustSet = robustSet;
const robustUnset = ({
  key,
  document
}) => {
  const splitKey = key.split('.');
  let tempKey = splitKey[0];
  for (let i = 0; i < splitKey.length - 1; i++) {
    if (i > 0) {
      tempKey += `.${splitKey[i]}`;
    }
    const value = document[tempKey];
    if (value != null) {
      if ((0, _is_objectlike_or_array_of_objectlikes.isObjectTypeGuard)(value)) {
        if (Object.keys(value).length !== 0) {
          robustUnset({
            key: splitKey.slice(i + 1).join('.'),
            document: value
          });
          // check if field was removed from object, if so, we remove empty parent too
          if (Object.keys(value).length === 0) {
            (0, _lodash.unset)(document, tempKey);
          }
        }
        return document;
      }
    }
  }
  (0, _lodash.unset)(document, key);
  return document;
};
exports.robustUnset = robustUnset;