"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getMaxClauseCountErrorValue = exports.getMatchedFields = exports.getFieldAndValueToDocIdsMap = exports.extractNamedQueries = exports.encodeThreatMatchNamedQuery = exports.decodeThreatMatchNamedQuery = exports.combineResults = exports.combineConcurrentResults = exports.calculateMaxLookBack = exports.calculateMax = exports.calculateAdditiveMax = exports.buildExecutionIntervalValidator = exports.MANY_NESTED_CLAUSES_ERR = exports.FAILED_CREATE_QUERY_MAX_CLAUSE = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _lodash = require("lodash");
var _types = require("../../../../telemetry/types");
var _utils = require("../../utils/utils");
var _types2 = require("./types");
var _check_error_details = require("../../utils/check_error_details");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MANY_NESTED_CLAUSES_ERR = exports.MANY_NESTED_CLAUSES_ERR = 'Query contains too many nested clauses; maxClauseCount is set to';
const FAILED_CREATE_QUERY_MAX_CLAUSE = exports.FAILED_CREATE_QUERY_MAX_CLAUSE = 'failed to create query: maxClauseCount is set to';

/**
 * Given two timers this will take the max of each and add them to each other and return that addition.
 * Max(timer_array_1) + Max(timer_array_2)
 * @param existingTimers String array of existing timers
 * @param newTimers String array of new timers.
 * @returns String array of the new maximum between the two timers
 */
const calculateAdditiveMax = (existingTimers, newTimers) => {
  const numericNewTimerMax = Math.max(0, ...newTimers.map(time => +time));
  const numericExistingTimerMax = Math.max(0, ...existingTimers.map(time => +time));
  return [String(numericNewTimerMax + numericExistingTimerMax)];
};

/**
 * Given two timers this will take the max of each and then get the max from each.
 * Max(Max(timer_array_1), Max(timer_array_2))
 * @param existingTimers String array of existing timers
 * @param newTimers String array of new timers.
 * @returns String array of the new maximum between the two timers
 */
exports.calculateAdditiveMax = calculateAdditiveMax;
const calculateMax = (existingTimers, newTimers) => {
  const numericNewTimerMax = Math.max(0, ...newTimers.map(time => +time));
  const numericExistingTimerMax = Math.max(0, ...existingTimers.map(time => +time));
  return String(Math.max(numericNewTimerMax, numericExistingTimerMax));
};

/**
 * Given two dates this will return the larger of the two unless one of them is null
 * or undefined. If both one or the other is null/undefined it will return the newDate.
 * If there is a mix of "undefined" and "null", this will prefer to set it to "null" as having
 * a higher value than "undefined"
 * @param existingDate The existing date which can be undefined or null or a date
 * @param newDate The new date which can be undefined or null or a date
 */
exports.calculateMax = calculateMax;
const calculateMaxLookBack = (existingDate, newDate) => {
  const newDateValue = newDate === null ? 1 : newDate === undefined ? 0 : newDate.valueOf();
  const existingDateValue = existingDate === null ? 1 : existingDate === undefined ? 0 : existingDate.valueOf();
  if (newDateValue >= existingDateValue) {
    return newDate;
  } else {
    return existingDate;
  }
};

/**
 * Combines two results together and returns the results combined
 * @param currentResult The current result to combine with a newResult
 * @param newResult The new result to combine
 */
exports.calculateMaxLookBack = calculateMaxLookBack;
const combineResults = (currentResult, newResult) => {
  var _currentResult$suppre, _newResult$suppressed;
  return {
    success: currentResult.success === false ? false : newResult.success,
    warning: currentResult.warning || newResult.warning,
    enrichmentTimes: calculateAdditiveMax(currentResult.enrichmentTimes, newResult.enrichmentTimes),
    bulkCreateTimes: calculateAdditiveMax(currentResult.bulkCreateTimes, newResult.bulkCreateTimes),
    searchAfterTimes: calculateAdditiveMax(currentResult.searchAfterTimes, newResult.searchAfterTimes),
    createdSignalsCount: currentResult.createdSignalsCount + newResult.createdSignalsCount,
    createdSignals: [...currentResult.createdSignals, ...newResult.createdSignals],
    warningMessages: [...currentResult.warningMessages, ...newResult.warningMessages],
    errors: [...new Set([...currentResult.errors, ...newResult.errors])],
    suppressedAlertsCount: ((_currentResult$suppre = currentResult.suppressedAlertsCount) !== null && _currentResult$suppre !== void 0 ? _currentResult$suppre : 0) + ((_newResult$suppressed = newResult.suppressedAlertsCount) !== null && _newResult$suppressed !== void 0 ? _newResult$suppressed : 0)
  };
};

/**
 * Combines two results together and returns the results combined
 * @param currentResult The current result to combine with a newResult
 * @param newResult The new result to combine
 */
exports.combineResults = combineResults;
const combineConcurrentResults = (currentResult, newResult) => {
  const maxedNewResult = newResult.reduce((accum, item) => {
    var _accum$suppressedAler, _item$suppressedAlert;
    const maxSearchAfterTime = calculateMax(accum.searchAfterTimes, item.searchAfterTimes);
    const maxEnrichmentTimes = calculateMax(accum.enrichmentTimes, item.enrichmentTimes);
    const maxBulkCreateTimes = calculateMax(accum.bulkCreateTimes, item.bulkCreateTimes);
    return {
      success: accum.success && item.success,
      warning: accum.warning || item.warning,
      searchAfterTimes: [maxSearchAfterTime],
      bulkCreateTimes: [maxBulkCreateTimes],
      enrichmentTimes: [maxEnrichmentTimes],
      createdSignalsCount: accum.createdSignalsCount + item.createdSignalsCount,
      createdSignals: [...accum.createdSignals, ...item.createdSignals],
      warningMessages: [...accum.warningMessages, ...item.warningMessages],
      errors: [...new Set([...accum.errors, ...item.errors])],
      userError: accum.userError || item.errors.every(err => (0, _check_error_details.checkErrorDetails)(err).isUserError),
      suppressedAlertsCount: ((_accum$suppressedAler = accum.suppressedAlertsCount) !== null && _accum$suppressedAler !== void 0 ? _accum$suppressedAler : 0) + ((_item$suppressedAlert = item.suppressedAlertsCount) !== null && _item$suppressedAlert !== void 0 ? _item$suppressedAlert : 0)
    };
  }, {
    success: true,
    userError: false,
    warning: false,
    searchAfterTimes: [],
    bulkCreateTimes: [],
    enrichmentTimes: [],
    createdSignalsCount: 0,
    suppressedAlertsCount: 0,
    createdSignals: [],
    errors: [],
    warningMessages: []
  });
  return combineResults(currentResult, maxedNewResult);
};
exports.combineConcurrentResults = combineConcurrentResults;
const separator = '__SEP__';
const encodeThreatMatchNamedQuery = query => {
  const {
    threatMappingIndex,
    queryType
  } = query;
  let id;
  let index;
  if ('id' in query) {
    id = query.id;
    index = query.index;
  }
  return [id, index, threatMappingIndex, queryType].join(separator);
};
exports.encodeThreatMatchNamedQuery = encodeThreatMatchNamedQuery;
const decodeThreatMatchNamedQuery = encoded => {
  const queryValues = encoded.split(separator);
  const [id, index, threatMappingIndexString, queryType] = queryValues;
  const threatMappingIndex = parseInt(threatMappingIndexString, 10);
  if (isNaN(threatMappingIndex)) {
    throw new Error(`Decoded threat mapping index is invalid. Decoded value: ${threatMappingIndexString}`);
  }
  const query = {
    id,
    index,
    threatMappingIndex,
    queryType
  };
  let isValidQuery = false;
  if (queryType === _types2.ThreatMatchQueryType.match) {
    isValidQuery = queryValues.length === 4 && queryValues.every(Boolean);
  }
  if (queryType === _types2.ThreatMatchQueryType.term) {
    // We checked if threatMappingIndex is a number above already, so at this point a decoded term query is valid
    isValidQuery = true;
  }
  if (!isValidQuery) {
    const queryString = JSON.stringify(query);
    throw new Error(`Decoded query is invalid. Decoded value: ${queryString}`);
  }
  return query;
};
exports.decodeThreatMatchNamedQuery = decodeThreatMatchNamedQuery;
const extractNamedQueries = hit => Array.isArray(hit.matched_queries) ? hit.matched_queries.map(match => decodeThreatMatchNamedQuery(match)) : [];
exports.extractNamedQueries = extractNamedQueries;
const buildExecutionIntervalValidator = interval => {
  const intervalDuration = (0, _utils.parseInterval)(interval);
  if (intervalDuration == null) {
    throw new Error(`Unable to parse rule interval (${interval}); stopping rule execution since allotted duration is undefined.`);
  }
  const executionEnd = (0, _moment.default)().add(intervalDuration);
  return () => {
    if ((0, _moment.default)().isAfter(executionEnd)) {
      const message = `Current rule execution has exceeded its allotted interval (${interval}) and has been stopped.`;
      throw new Error(message);
    }
  };
};

/*
 * Return list of fields by type used for matching in IM rule
 */
exports.buildExecutionIntervalValidator = buildExecutionIntervalValidator;
const getMatchedFields = threatMapping => threatMapping.reduce((acc, val) => {
  val.entries.forEach(mapping => {
    if (!acc.source.includes(mapping.field)) {
      acc.source.push(mapping.field);
    }
    if (!acc.threat.includes(mapping.value)) {
      acc.threat.push(mapping.value);
    }
  });
  return acc;
}, {
  source: [],
  threat: []
});
exports.getMatchedFields = getMatchedFields;
const getFieldAndValueToDocIdsMap = ({
  eventList,
  threatMatchedFields
}) => eventList.reduce((acc, event) => {
  threatMatchedFields.source.forEach(field => {
    var _get;
    const fieldValue = (_get = (0, _lodash.get)(event.fields, field)) === null || _get === void 0 ? void 0 : _get[0];
    if (!fieldValue) return;
    if (!acc[field]) {
      acc[field] = {};
    }
    if (!acc[field][fieldValue]) {
      acc[field][fieldValue] = [];
    }
    // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
    acc[field][fieldValue].push(event._id);
  });
  return acc;
}, {});
exports.getFieldAndValueToDocIdsMap = getFieldAndValueToDocIdsMap;
const getMaxClauseCountErrorValue = (searchesPerformed, threatEntriesCount, previousChunkSize, eventsTelemetry) => searchesPerformed.reduce((acc, search) => {
  var _failedToCreateQueryM, _tooManyNestedClauses;
  const failedToCreateQueryMessage = search.errors.find(err => err.includes(FAILED_CREATE_QUERY_MAX_CLAUSE));

  // the below error is specific to an error returned by getSignalsQueryMapFromThreatIndex
  const tooManyNestedClausesMessage = search.errors.find(err => err.includes(MANY_NESTED_CLAUSES_ERR));
  const regex = /[0-9]+/g;
  const foundMaxClauseCountValue = failedToCreateQueryMessage === null || failedToCreateQueryMessage === void 0 ? void 0 : (_failedToCreateQueryM = failedToCreateQueryMessage.match(regex)) === null || _failedToCreateQueryM === void 0 ? void 0 : _failedToCreateQueryM[0];
  const foundNestedClauseCountValue = tooManyNestedClausesMessage === null || tooManyNestedClausesMessage === void 0 ? void 0 : (_tooManyNestedClauses = tooManyNestedClausesMessage.match(regex)) === null || _tooManyNestedClauses === void 0 ? void 0 : _tooManyNestedClauses[0];
  if (foundNestedClauseCountValue != null && !(0, _lodash.isEmpty)(foundNestedClauseCountValue)) {
    const errorType = `${MANY_NESTED_CLAUSES_ERR} ${foundNestedClauseCountValue}`;
    const tempVal = parseInt(foundNestedClauseCountValue, 10);
    eventsTelemetry === null || eventsTelemetry === void 0 ? void 0 : eventsTelemetry.sendAsync(_types.TelemetryChannel.DETECTION_ALERTS, [`Query contains too many nested clauses error received during IM search`]);

    // minus 1 since the max clause count value is exclusive
    // multiplying by two because we need to account for the
    // threat fields and event fields. A single threat entries count
    // is comprised of two fields, one field from the threat index
    // and another field from the event index. so we need to multiply by 2
    // to cover the fact that the nested clause error happens
    // because we are searching over event and threat fields.
    // so we need to make this smaller than a single 'failed to create query'
    // max clause count error.
    const val = Math.floor((tempVal - 1) / (2 * (threatEntriesCount + 1)));
    // There is a chance the new calculated val still may yield a too many nested queries
    // error message. In that case we want to make sure we don't fall into an infinite loop
    // and so we send a new value that is guaranteed to be smaller than the previous one.
    if (val >= previousChunkSize) {
      return {
        maxClauseCountValue: Math.floor(previousChunkSize / 2),
        errorType
      };
    }
    return {
      maxClauseCountValue: val,
      errorType
    };
  } else if (foundMaxClauseCountValue != null && !(0, _lodash.isEmpty)(foundMaxClauseCountValue)) {
    const errorType = `${FAILED_CREATE_QUERY_MAX_CLAUSE} ${foundNestedClauseCountValue}`;
    const tempVal = parseInt(foundMaxClauseCountValue, 10);
    eventsTelemetry === null || eventsTelemetry === void 0 ? void 0 : eventsTelemetry.sendAsync(_types.TelemetryChannel.DETECTION_ALERTS, [`failed to create query error received during IM search`]);
    // minus 1 since the max clause count value is exclusive
    // and we add 1 to threatEntries to increase the number of "buckets"
    // that our searches are spread over, smaller buckets means less clauses
    const val = Math.floor((tempVal - 1) / (threatEntriesCount + 1));
    return {
      maxClauseCountValue: val,
      errorType
    };
  } else {
    return acc;
  }
}, {
  maxClauseCountValue: Number.NEGATIVE_INFINITY,
  errorType: 'no helpful error message available'
});
exports.getMaxClauseCountErrorValue = getMaxClauseCountErrorValue;