"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bulkCreate = void 0;
var _perf_hooks = require("perf_hooks");
var _lodash = require("lodash");
var _utils = require("../utils/utils");
var _enrichments = require("../utils/enrichments");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const bulkCreate = async ({
  wrappedAlerts,
  services,
  sharedParams,
  maxAlerts
}) => {
  const {
    ruleExecutionLogger,
    refreshOnIndexingAlerts: refreshForBulkCreate
  } = sharedParams;
  const {
    alertWithPersistence
  } = services;
  if (wrappedAlerts.length === 0) {
    return {
      errors: [],
      success: true,
      enrichmentDuration: '0',
      bulkCreateDuration: '0',
      createdItemsCount: 0,
      createdItems: [],
      alertsWereTruncated: false
    };
  }
  const start = _perf_hooks.performance.now();
  let enrichmentsTimeStart = 0;
  let enrichmentsTimeFinish = 0;
  const enrichAlertsWrapper = async (alerts, params) => {
    enrichmentsTimeStart = _perf_hooks.performance.now();
    try {
      const enrichedAlerts = await (0, _enrichments.enrichEvents)({
        services,
        logger: sharedParams.ruleExecutionLogger,
        events: alerts,
        spaceId: params.spaceId
      });
      return enrichedAlerts;
    } catch (error) {
      ruleExecutionLogger.error(`Alerts enrichment failed: ${error}`);
      throw error;
    } finally {
      enrichmentsTimeFinish = _perf_hooks.performance.now();
    }
  };
  const {
    createdAlerts,
    errors,
    alertsWereTruncated
  } = await alertWithPersistence(wrappedAlerts.map(doc => ({
    _id: doc._id,
    // `fields` should have already been merged into `doc._source`
    _source: doc._source
  })), refreshForBulkCreate, maxAlerts, enrichAlertsWrapper);
  const end = _perf_hooks.performance.now();
  ruleExecutionLogger.debug(`Alerts bulk process took ${(0, _utils.makeFloatString)(end - start)} ms`);
  if (!(0, _lodash.isEmpty)(errors)) {
    ruleExecutionLogger.warn(`Alerts bulk process finished with errors: ${JSON.stringify(errors)}`);
    return {
      errors: Object.keys(errors),
      success: false,
      enrichmentDuration: (0, _utils.makeFloatString)(enrichmentsTimeFinish - enrichmentsTimeStart),
      bulkCreateDuration: (0, _utils.makeFloatString)(end - start),
      createdItemsCount: createdAlerts.length,
      createdItems: createdAlerts,
      alertsWereTruncated
    };
  } else {
    return {
      errors: [],
      success: true,
      bulkCreateDuration: (0, _utils.makeFloatString)(end - start),
      enrichmentDuration: (0, _utils.makeFloatString)(enrichmentsTimeFinish - enrichmentsTimeStart),
      createdItemsCount: createdAlerts.length,
      createdItems: createdAlerts,
      alertsWereTruncated
    };
  }
};
exports.bulkCreate = bulkCreate;