"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.performEsqlRequest = void 0;
var _perf_hooks = require("perf_hooks");
var _server = require("@kbn/kibana-utils-plugin/server");
var _logged_requests = require("../utils/logged_requests");
var i18n = _interopRequireWildcard(require("../translations"));
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const setLatestRequestDuration = (startTime, loggedRequests) => {
  const duration = _perf_hooks.performance.now() - startTime;
  if (loggedRequests && loggedRequests !== null && loggedRequests !== void 0 && loggedRequests[loggedRequests.length - 1]) {
    loggedRequests[loggedRequests.length - 1].duration = Math.round(duration);
  }
};
const performEsqlRequest = async ({
  esClient,
  requestBody,
  requestQueryParams,
  ruleExecutionLogger,
  shouldStopExecution,
  loggedRequests
}) => {
  let pollInterval = 10 * 1000; // Poll every 10 seconds
  let pollCount = 0;
  let queryId = '';
  try {
    loggedRequests === null || loggedRequests === void 0 ? void 0 : loggedRequests.push({
      request: (0, _logged_requests.logEsqlRequest)(requestBody, requestQueryParams),
      description: i18n.ESQL_SEARCH_REQUEST_DESCRIPTION,
      request_type: 'findMatches'
    });
    const asyncSearchStarted = _perf_hooks.performance.now();
    const asyncEsqlResponse = await esClient.transport.request({
      method: 'POST',
      path: '/_query/async',
      body: requestBody,
      querystring: requestQueryParams
    });
    setLatestRequestDuration(asyncSearchStarted, loggedRequests);
    queryId = asyncEsqlResponse.id;
    const isRunning = asyncEsqlResponse.is_running;
    if (!isRunning) {
      return asyncEsqlResponse;
    }

    // Poll for long-executing query
    while (true) {
      await new Promise(resolve => setTimeout(resolve, pollInterval));
      loggedRequests === null || loggedRequests === void 0 ? void 0 : loggedRequests.push({
        request: `GET /_query/async/${queryId}`,
        description: i18n.ESQL_POLL_REQUEST_DESCRIPTION
      });
      const pollStarted = _perf_hooks.performance.now();
      const pollResponse = await esClient.transport.request({
        method: 'GET',
        path: `/_query/async/${queryId}`
      });
      setLatestRequestDuration(pollStarted, loggedRequests);
      if (!pollResponse.is_running) {
        return pollResponse;
      }
      pollCount++;
      if (pollCount > 60) {
        pollInterval = 60 * 1000; // Increase the poll interval after 10m
      }
      const isCancelled = shouldStopExecution(); // Execution will be cancelled if rule times out
      ruleExecutionLogger === null || ruleExecutionLogger === void 0 ? void 0 : ruleExecutionLogger.debug(`Polling for query ID: ${queryId}, isCancelled: ${isCancelled}`);
      if (isCancelled) {
        throw new Error('Rule execution cancelled due to timeout');
      }
      ruleExecutionLogger === null || ruleExecutionLogger === void 0 ? void 0 : ruleExecutionLogger.debug(`Query is still running for query ID: ${queryId}`);
    }
  } catch (error) {
    ruleExecutionLogger === null || ruleExecutionLogger === void 0 ? void 0 : ruleExecutionLogger.error(`Error while performing ES|QL search: ${error === null || error === void 0 ? void 0 : error.message}`);
    throw (0, _server.getKbnServerError)(error);
  } finally {
    if (queryId) {
      loggedRequests === null || loggedRequests === void 0 ? void 0 : loggedRequests.push({
        request: `DELETE /_query/async/${queryId}`,
        description: i18n.ESQL_DELETE_REQUEST_DESCRIPTION
      });
      const deleteStarted = _perf_hooks.performance.now();
      await esClient.transport.request({
        method: 'DELETE',
        path: `/_query/async/${queryId}`
      });
      setLatestRequestDuration(deleteStarted, loggedRequests);
    }
  }
};
exports.performEsqlRequest = performEsqlRequest;