"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.normalizeSpacesAggregation = exports.getSpacesAggregation = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// The number of Kibana spaces is limited by the `maxSpaces` config setting of the spaces plugin.
// At the time of writing this comment, `maxSpaces` defaults to 1000.
// So in normal conditions there can't exist more than 1000 Kibana spaces.
//
// However, we set `MAX_KIBANA_SPACES` to a higher value to handle rare cases when there are more
// than 1000 spaces in a cluster. Hopefully it will cover 99.(9)% of use cases.
//
// In the rest of the edge cases, we will be missing some spaces, but the effect of this will be
// limited by the fact that the aggregation below will sort spaces desc by rules count in them.
// It will return spaces with most of the existing rules, and will not return spaces with fewer
// number of rules. Hopefully, we will miss only spaces with very few rules. This should be
// acceptable because the goal of getting all space ids in the rule monitoring subdomain is to be
// able to aggregate health metrics for those spaces. It's unlikely that spaces with very few rules
// will have a major impact on health and performance metrics of the whole cluster.
const MAX_KIBANA_SPACES = 10_000;
const getSpacesAggregation = () => {
  return {
    rulesBySpace: {
      terms: {
        field: 'alert.namespaces',
        size: MAX_KIBANA_SPACES
      }
    }
  };
};
exports.getSpacesAggregation = getSpacesAggregation;
const normalizeSpacesAggregation = aggregations => {
  const rulesBySpace = (aggregations === null || aggregations === void 0 ? void 0 : aggregations.rulesBySpace) || {};
  const buckets = rulesBySpace.buckets || [];
  const spaceIds = buckets.map(b => String(b.key));
  return spaceIds;
};
exports.normalizeSpacesAggregation = normalizeSpacesAggregation;