"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateRuleRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _zodHelpers = require("@kbn/zod-helpers");
var _rule_management = require("../../../../../../../common/api/detection_engine/rule_management");
var _constants = require("../../../../../../../common/constants");
var _utils = require("../../../../routes/utils");
var _read_rules = require("../../../logic/detection_rules_client/read_rules");
var _check_for_default_rule_exception_list = require("../../../logic/exceptions/check_for_default_rule_exception_list");
var _validate_rule_default_exception_list = require("../../../logic/exceptions/validate_rule_default_exception_list");
var _utils2 = require("../../../utils/utils");
var _validate = require("../../../utils/validate");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const updateRuleRoute = router => {
  router.versioned.put({
    access: 'public',
    path: _constants.DETECTION_ENGINE_RULES_URL,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: {
        body: (0, _zodHelpers.buildRouteValidationWithZod)(_rule_management.UpdateRuleRequestBody)
      }
    }
  }, async (context, request, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    const validationErrors = (0, _rule_management.validateUpdateRuleProps)(request.body);
    if (validationErrors.length) {
      return siemResponse.error({
        statusCode: 400,
        body: validationErrors
      });
    }
    try {
      const ctx = await context.resolve(['core', 'securitySolution', 'alerting', 'licensing']);
      const rulesClient = await ctx.alerting.getRulesClient();
      const detectionRulesClient = ctx.securitySolution.getDetectionRulesClient();
      (0, _check_for_default_rule_exception_list.checkDefaultRuleExceptionListReferences)({
        exceptionLists: request.body.exceptions_list
      });
      await (0, _validate_rule_default_exception_list.validateRuleDefaultExceptionList)({
        exceptionsList: request.body.exceptions_list,
        rulesClient,
        ruleRuleId: request.body.rule_id,
        ruleId: request.body.id
      });
      const existingRule = await (0, _read_rules.readRules)({
        rulesClient,
        ruleId: request.body.rule_id,
        id: request.body.id
      });
      if (existingRule == null) {
        const error = (0, _utils2.getIdError)({
          id: request.body.id,
          ruleId: request.body.rule_id
        });
        return siemResponse.error({
          body: error.message,
          statusCode: error.statusCode
        });
      }
      await (0, _validate.validateResponseActionsPermissions)(ctx.securitySolution, request.body, existingRule);
      const updatedRule = await detectionRulesClient.updateRule({
        ruleUpdate: request.body
      });
      return response.ok({
        body: updatedRule
      });
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return siemResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.updateRuleRoute = updateRuleRoute;