"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createPrebuiltRuleObjectsClient = void 0;
var _with_security_span = require("../../../../../utils/with_security_span");
var _find_rules = require("../../../rule_management/logic/search/find_rules");
var _internal_rule_to_api_response = require("../../../rule_management/logic/detection_rules_client/converters/internal_rule_to_api_response");
var _rule_filtering = require("../../../../../../common/detection_engine/rule_management/rule_filtering");
var _get_existing_prepackaged_rules = require("../../../rule_management/logic/search/get_existing_prepackaged_rules");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createPrebuiltRuleObjectsClient = rulesClient => {
  return {
    fetchInstalledRulesByIds: ({
      ruleIds,
      sortField = 'createdAt',
      sortOrder = 'desc'
    }) => {
      return (0, _with_security_span.withSecuritySpan)('IPrebuiltRuleObjectsClient.fetchInstalledRulesByIds', async () => {
        if (ruleIds.length === 0) {
          return [];
        }
        const {
          data
        } = await (0, _find_rules.findRules)({
          rulesClient,
          perPage: ruleIds.length,
          page: 1,
          sortField,
          sortOrder,
          fields: undefined,
          filter: `alert.attributes.params.ruleId:(${ruleIds.join(' or ')})`
        });
        const rules = data.map(rule => (0, _internal_rule_to_api_response.internalRuleToAPIResponse)(rule));
        return rules;
      });
    },
    fetchInstalledRules: ({
      page,
      perPage,
      sortField,
      sortOrder,
      filter
    } = {}) => {
      return (0, _with_security_span.withSecuritySpan)('IPrebuiltRuleObjectsClient.fetchInstalledRules', async () => {
        const filterKQL = (0, _rule_filtering.convertRulesFilterToKQL)({
          showElasticRules: true,
          filter: filter === null || filter === void 0 ? void 0 : filter.name,
          tags: filter === null || filter === void 0 ? void 0 : filter.tags,
          customizationStatus: filter === null || filter === void 0 ? void 0 : filter.customization_status
        });
        const rulesData = await (0, _find_rules.findRules)({
          rulesClient,
          filter: filterKQL,
          perPage,
          page,
          sortField,
          sortOrder,
          fields: undefined
        });
        const rules = rulesData.data.map(rule => (0, _internal_rule_to_api_response.internalRuleToAPIResponse)(rule));
        return rules;
      });
    },
    fetchInstalledRuleVersionsByIds: ({
      ruleIds,
      sortField,
      sortOrder
    }) => {
      return (0, _with_security_span.withSecuritySpan)('IPrebuiltRuleObjectsClient.fetchInstalledRuleVersionsByIds', async () => {
        const filterKQL = (0, _rule_filtering.convertRulesFilterToKQL)({
          showElasticRules: true
        });
        const rulesData = await (0, _find_rules.findRules)({
          rulesClient,
          ruleIds,
          filter: filterKQL,
          perPage: _get_existing_prepackaged_rules.MAX_PREBUILT_RULES_COUNT,
          page: 1,
          sortField,
          sortOrder,
          fields: ['params.ruleId', 'params.version', 'tags']
        });
        return rulesData.data.map(rule => ({
          id: rule.id,
          rule_id: rule.params.ruleId,
          version: rule.params.version,
          tags: rule.tags
        }));
      });
    },
    fetchInstalledRuleVersions: ({
      filter,
      sortField,
      sortOrder
    } = {}) => {
      return (0, _with_security_span.withSecuritySpan)('IPrebuiltRuleObjectsClient.fetchInstalledRuleVersions', async () => {
        const filterKQL = (0, _rule_filtering.convertRulesFilterToKQL)({
          showElasticRules: true,
          filter: filter === null || filter === void 0 ? void 0 : filter.name,
          tags: filter === null || filter === void 0 ? void 0 : filter.tags,
          customizationStatus: filter === null || filter === void 0 ? void 0 : filter.customization_status
        });
        const rulesData = await (0, _find_rules.findRules)({
          rulesClient,
          filter: filterKQL,
          perPage: _get_existing_prepackaged_rules.MAX_PREBUILT_RULES_COUNT,
          page: 1,
          sortField,
          sortOrder,
          fields: ['params.ruleId', 'params.version', 'tags']
        });
        return rulesData.data.map(rule => ({
          id: rule.id,
          rule_id: rule.params.ruleId,
          version: rule.params.version,
          tags: rule.tags
        }));
      });
    }
  };
};
exports.createPrebuiltRuleObjectsClient = createPrebuiltRuleObjectsClient;