"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.dataSourceDiffAlgorithm = void 0;
var _utility_types = require("../../../../../../../../common/utility_types");
var _prebuilt_rules = require("../../../../../../../../common/api/detection_engine/prebuilt_rules");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Takes a type of `RuleDataSource | undefined` because the data source can be index patterns, a data view id, or neither in some cases
 */
const dataSourceDiffAlgorithm = (versions, isRuleCustomized) => {
  const {
    base_version: baseVersion,
    current_version: currentVersion,
    target_version: targetVersion
  } = versions;
  const diffOutcome = (0, _prebuilt_rules.determineDiffOutcomeForDataSource)(baseVersion, currentVersion, targetVersion);
  const valueCanUpdate = (0, _prebuilt_rules.determineIfValueCanUpdate)(diffOutcome);
  const hasBaseVersion = baseVersion !== _prebuilt_rules.MissingVersion;
  const {
    mergeOutcome,
    conflict,
    mergedVersion
  } = mergeVersions({
    baseVersion: hasBaseVersion ? baseVersion : undefined,
    currentVersion,
    targetVersion,
    diffOutcome,
    isRuleCustomized
  });
  return {
    has_base_version: hasBaseVersion,
    base_version: hasBaseVersion ? baseVersion : undefined,
    current_version: currentVersion,
    target_version: targetVersion,
    merged_version: mergedVersion,
    merge_outcome: mergeOutcome,
    diff_outcome: diffOutcome,
    conflict,
    has_update: valueCanUpdate
  };
};
exports.dataSourceDiffAlgorithm = dataSourceDiffAlgorithm;
const mergeVersions = ({
  baseVersion,
  currentVersion,
  targetVersion,
  diffOutcome,
  isRuleCustomized
}) => {
  const dedupedBaseVersion = baseVersion ? (0, _helpers.getDedupedDataSourceVersion)(baseVersion) : baseVersion;
  const dedupedCurrentVersion = currentVersion ? (0, _helpers.getDedupedDataSourceVersion)(currentVersion) : currentVersion;
  const dedupedTargetVersion = targetVersion ? (0, _helpers.getDedupedDataSourceVersion)(targetVersion) : targetVersion;
  switch (diffOutcome) {
    case _prebuilt_rules.ThreeWayDiffOutcome.StockValueNoUpdate:
    case _prebuilt_rules.ThreeWayDiffOutcome.CustomizedValueNoUpdate:
    case _prebuilt_rules.ThreeWayDiffOutcome.CustomizedValueSameUpdate:
      return {
        conflict: _prebuilt_rules.ThreeWayDiffConflict.NONE,
        mergeOutcome: _prebuilt_rules.ThreeWayMergeOutcome.Current,
        mergedVersion: dedupedCurrentVersion
      };
    case _prebuilt_rules.ThreeWayDiffOutcome.StockValueCanUpdate:
      return {
        conflict: _prebuilt_rules.ThreeWayDiffConflict.NONE,
        mergeOutcome: _prebuilt_rules.ThreeWayMergeOutcome.Target,
        mergedVersion: dedupedTargetVersion
      };
    case _prebuilt_rules.ThreeWayDiffOutcome.CustomizedValueCanUpdate:
      {
        if ((0, _prebuilt_rules.isIndexPatternDataSourceType)(dedupedCurrentVersion) && (0, _prebuilt_rules.isIndexPatternDataSourceType)(dedupedTargetVersion)) {
          const baseVersionToMerge = dedupedBaseVersion && dedupedBaseVersion.type === _prebuilt_rules.DataSourceType.index_patterns ? dedupedBaseVersion.index_patterns : [];
          return {
            conflict: _prebuilt_rules.ThreeWayDiffConflict.SOLVABLE,
            mergeOutcome: _prebuilt_rules.ThreeWayMergeOutcome.Merged,
            mergedVersion: {
              type: _prebuilt_rules.DataSourceType.index_patterns,
              index_patterns: (0, _helpers.mergeDedupedArrays)(baseVersionToMerge, dedupedCurrentVersion.index_patterns, dedupedTargetVersion.index_patterns)
            }
          };
        }
        return {
          conflict: _prebuilt_rules.ThreeWayDiffConflict.NON_SOLVABLE,
          mergeOutcome: _prebuilt_rules.ThreeWayMergeOutcome.Current,
          mergedVersion: dedupedCurrentVersion
        };
      }

    // Missing base versions always return target version
    // Scenario -AA is treated as AAA
    // https://github.com/elastic/kibana/issues/210358#issuecomment-2654492854
    case _prebuilt_rules.ThreeWayDiffOutcome.MissingBaseNoUpdate:
      {
        return {
          conflict: _prebuilt_rules.ThreeWayDiffConflict.NONE,
          mergedVersion: dedupedTargetVersion,
          mergeOutcome: _prebuilt_rules.ThreeWayMergeOutcome.Target
        };
      }

    // Missing base versions always return target version
    // If the rule is customized, we return a SOLVABLE conflict
    // Otherwise we treat scenario -AB as AAB
    // https://github.com/elastic/kibana/issues/210358#issuecomment-2654492854
    case _prebuilt_rules.ThreeWayDiffOutcome.MissingBaseCanUpdate:
      {
        return {
          conflict: isRuleCustomized ? _prebuilt_rules.ThreeWayDiffConflict.SOLVABLE : _prebuilt_rules.ThreeWayDiffConflict.NONE,
          mergedVersion: dedupedTargetVersion,
          mergeOutcome: _prebuilt_rules.ThreeWayMergeOutcome.Target
        };
      }
    default:
      return (0, _utility_types.assertUnreachable)(diffOutcome);
  }
};