"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAnalyzeCompressedIndexMappingAgent = void 0;
var _zod = require("@kbn/zod");
var _langgraph = require("@langchain/langgraph");
var _messages = require("@langchain/core/messages");
var _inspect_index_utils = require("../../../../tools/inspect_index_mapping_tool/inspect_index_utils");
var _compress_mapping = require("./compress_mapping");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const structuredOutput = _zod.z.object({
  containsRequiredFieldsForQuery: _zod.z.boolean().describe('Whether the index pattern contains the required fields for the query')
});
const getAnalyzeCompressedIndexMappingAgent = async ({
  createLlmInstance
}) => {
  const llm = await createLlmInstance();
  return async state => {
    const {
      fieldDescriptors,
      input
    } = state;
    if (fieldDescriptors === undefined) {
      throw new Error('State fieldDescriptors is undefined');
    }
    if (input === undefined) {
      throw new Error('State input is undefined');
    }
    const prunedFields = fieldDescriptors.map(fieldDescriptor => ({
      name: fieldDescriptor.name,
      type: fieldDescriptor.esTypes[0]
    }));
    const nestedObject = (0, _inspect_index_utils.mapFieldDescriptorToNestedObject)(prunedFields);
    const compressedIndexMapping = (0, _compress_mapping.compressMapping)(nestedObject);
    const result = await llm.withStructuredOutput(structuredOutput, {
      name: 'indexMappingAnalysis'
    }).invoke([new _messages.SystemMessage({
      content: 'You are a security analyst who is an expert in Elasticsearch and particularly at analyzing indices. ' + 'You will be given an compressed index mapping containing available fields and types and an explanation ' + 'of the query that we are trying to generate. Analyze the index mapping and determine whether it contains the ' + 'fields required to write the query. You do not need to generate the query right now, just determine whether the' + ' index mapping contains the fields required to write the query.'
    }), new _messages.HumanMessage({
      content: `Query objective:\n'${input.question}'\n\nIndex pattern:\n'${input.indexPattern}'\n\nCompressed index mapping:\n${compressedIndexMapping}`
    })]);
    return new _langgraph.Command({
      update: {
        output: {
          containsRequiredFieldsForQuery: result.containsRequiredFieldsForQuery,
          context: compressedIndexMapping
        }
      }
    });
  };
};
exports.getAnalyzeCompressedIndexMappingAgent = getAnalyzeCompressedIndexMappingAgent;