"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.add = add;
exports.angle = angle;
exports.applyMatrix3 = applyMatrix3;
exports.clamp = clamp;
exports.distance = distance;
exports.divide = divide;
exports.isEqual = isEqual;
exports.length = length;
exports.lerp = lerp;
exports.multiply = multiply;
exports.scale = scale;
exports.subtract = subtract;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Returns a vector which is the sum of `a` and `b`.
 */
function add(a, b) {
  return [a[0] + b[0], a[1] + b[1]];
}

/**
 * Returns a vector which is the difference of `a` and `b`.
 */
function subtract(a, b) {
  return [a[0] - b[0], a[1] - b[1]];
}

/**
 * Returns a vector which is the quotient of `a` and `b`.
 */
function divide(a, b) {
  return [a[0] / b[0], a[1] / b[1]];
}

/**
 * Return `[ a[0] * b[0], a[1] * b[1] ]`
 */
function multiply(a, b) {
  return [a[0] * b[0], a[1] * b[1]];
}

/**
 * Returns a vector which is the result of applying a 2D transformation matrix to the provided vector.
 */
function applyMatrix3([x, y], [m11, m12, m13, m21, m22, m23]) {
  return [x * m11 + y * m12 + m13, x * m21 + y * m22 + m23];
}

/**
 * Returns a boolean indicating equality of two vectors.
 */
function isEqual([x1, y1], [x2, y2]) {
  return x1 === x2 && y1 === y2;
}

/**
 * Returns the distance between two vectors
 */
function distance(a, b) {
  const [x1, y1] = a;
  const [x2, y2] = b;
  return Math.sqrt(Math.pow(x2 - x1, 2) + Math.pow(y2 - y1, 2));
}

/**
 * Returns the angle between two vectors
 */
function angle(a, b) {
  const deltaX = b[0] - a[0];
  const deltaY = b[1] - a[1];
  return Math.atan2(deltaY, deltaX);
}

/**
 * Clamp `vector`'s components.
 */
function clamp([x, y], [minX, minY], [maxX, maxY]) {
  return [Math.max(minX, Math.min(maxX, x)), Math.max(minY, Math.min(maxY, y))];
}

/**
 * Scale vector by number
 */
function scale(a, n) {
  return [a[0] * n, a[1] * n];
}

/**
 * Linearly interpolate between `a` and `b`.
 * `t` represents progress and:
 * 0 <= `t` <= 1
 */
function lerp(a, b, t) {
  return add(scale(a, 1 - t), scale(b, t));
}

/**
 * The length of the vector
 */
function length([x, y]) {
  return Math.sqrt(x * x + y * y);
}