"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useOnboardingTelemetry = void 0;
var _react = require("react");
var _kibana_react = require("../../common/lib/kibana/kibana_react");
var _constants = require("../constants");
var _telemetry = require("../../common/lib/telemetry");
var _config = require("../config");
var _telemetry2 = require("./lib/telemetry");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useOnboardingTelemetry = () => {
  const {
    telemetry
  } = (0, _kibana_react.useKibana)().services;
  return (0, _react.useMemo)(() => ({
    reportCardOpen: (cardId, {
      auto = false
    } = {}) => {
      telemetry.reportEvent(_telemetry.OnboardingHubEventTypes.OnboardingHubStepOpen, {
        stepId: getStepId(cardId),
        trigger: auto ? 'navigation' : 'click'
      });
    },
    reportCardComplete: (cardId, {
      auto = false
    } = {}) => {
      telemetry.reportEvent(_telemetry.OnboardingHubEventTypes.OnboardingHubStepFinished, {
        stepId: getStepId(cardId),
        trigger: auto ? 'auto_check' : 'click'
      });
    },
    reportCardLinkClicked: (cardId, linkId) => {
      telemetry.reportEvent(_telemetry.OnboardingHubEventTypes.OnboardingHubStepLinkClicked, {
        originStepId: getStepId(cardId),
        stepLinkId: linkId
      });
    },
    reportCardSelectorClicked: (cardId, selectorId) => {
      telemetry.reportEvent(_telemetry.OnboardingHubEventTypes.OnboardingHubStepSelectorClicked, {
        originStepId: getStepId(cardId),
        selectorId
      });
    },
    reportLinkClick: _telemetry2.trackOnboardingLinkClick
  }), [telemetry]);
};

/**
 * Get the step id for a given card id.
 * The stepId is used to track the onboarding card in telemetry, it is a combination of the topic id and the card id.
 * To keep backwards compatibility, if the card is in the default topic, the stepId will be the card id only.
 */
exports.useOnboardingTelemetry = useOnboardingTelemetry;
const getStepId = cardId => {
  const cardTopic = _config.onboardingConfig.find(topic => topic.body.some(group => group.cards.some(card => card.id === cardId)));
  if (!cardTopic || cardTopic.id === _constants.OnboardingTopicId.default) {
    return cardId; // Do not add topic id for default topic to preserve existing events format
  }
  return `${cardTopic.id}#${cardId}`;
};