"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useUrlDetail = exports.useSyncUrlDetails = exports.getCardIdFromHash = void 0;
var _react = require("react");
var _securitySolutionNavigation = require("@kbn/security-solution-navigation");
var _use_stored_state = require("./use_stored_state");
var _constants = require("../../constants");
var _onboarding_context = require("../onboarding_context");
var _use_topic_id = require("./use_topic_id");
var _use_cloud_topic_id = require("./use_cloud_topic_id");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getCardIdFromHash = hash => hash.split('?')[0].replace('#', '') || null;
exports.getCardIdFromHash = getCardIdFromHash;
const setHash = cardId => {
  history.replaceState(null, '', cardId == null ? ' ' : `#${cardId}`);
};
const getUrlDetail = (topicId, cardId) => {
  return `${topicId !== _constants.OnboardingTopicId.default ? topicId : ''}${cardId ? `#${cardId}` : ''}`;
};

/**
 * This hook manages the expanded card id state in the LocalStorage and the hash in the URL.
 * The "urlDetail" is the combination of the topicId as the path fragment followed cardId in the hash (#) parameter, in the URL
 */
const useUrlDetail = () => {
  const {
    spaceId,
    telemetry
  } = (0, _onboarding_context.useOnboardingContext)();
  const topicId = (0, _use_topic_id.useTopicId)();
  const [storedUrlDetail, setStoredUrlDetail] = (0, _use_stored_state.useStoredUrlDetails)(spaceId);
  const {
    navigateTo
  } = (0, _securitySolutionNavigation.useNavigateTo)();
  const navigateToDetail = (0, _react.useCallback)(detail => {
    navigateTo({
      deepLinkId: _securitySolutionNavigation.SecurityPageName.landing,
      path: detail || undefined
    });
  }, [navigateTo]);
  const setTopic = (0, _react.useCallback)(newTopicId => {
    const detail = newTopicId === _constants.OnboardingTopicId.default ? null : newTopicId;
    setStoredUrlDetail(detail);
    navigateToDetail(detail);
  }, [setStoredUrlDetail, navigateToDetail]);
  const setCard = (0, _react.useCallback)(newCardId => {
    setHash(newCardId);
    setStoredUrlDetail(getUrlDetail(topicId, newCardId) || null);
    if (newCardId != null) {
      telemetry.reportCardOpen(newCardId);
    }
  }, [setStoredUrlDetail, topicId, telemetry]);
  return {
    topicId,
    setTopic,
    setCard,
    navigateToDetail,
    storedUrlDetail,
    setStoredUrlDetail
  };
};
exports.useUrlDetail = useUrlDetail;
/**
 * This hook manages the expanded card id state in the LocalStorage and the hash in the URL.
 */
const useSyncUrlDetails = ({
  pathTopicId,
  hashCardId
}) => {
  const {
    config,
    telemetry
  } = (0, _onboarding_context.useOnboardingContext)();
  const {
    storedUrlDetail,
    setStoredUrlDetail,
    navigateToDetail,
    setTopic
  } = useUrlDetail();
  const onComplete = (0, _react.useCallback)(cloudTopicId => {
    if (cloudTopicId && config.has(cloudTopicId)) {
      setTopic(cloudTopicId);
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);
  const {
    start: startGetCloudTopicId,
    isLoading
  } = (0, _use_cloud_topic_id.useCloudTopicId)({
    onComplete
  });
  (0, _react.useEffect)(() => {
    // Create the URL detail
    const urlDetail = `${pathTopicId || ''}${hashCardId ? `#${hashCardId}` : ''}`;

    // If the URL has a topic it has prevalence, we need to set it to the local storage
    if (urlDetail && urlDetail !== storedUrlDetail) {
      if (hashCardId) {
        telemetry.reportCardOpen(hashCardId, {
          auto: true
        });
      }
      setStoredUrlDetail(urlDetail);
      return;
    }

    // If the URL has no topic, but the local storage has a topic, we need to navigate to the topic
    if (!urlDetail && storedUrlDetail) {
      // Check if the stored topic is not valid, if so clear it to prevent inconsistencies
      const [storedTopicId] = storedUrlDetail.split('#');
      if (storedTopicId && !config.has(storedTopicId)) {
        setStoredUrlDetail(null);
        return;
      }
      navigateToDetail(storedUrlDetail);
    }

    // If nothing is stored and nothing is in the URL, let's see if we have a cloud topic (first time onboarding)
    if (!urlDetail && storedUrlDetail === undefined) {
      startGetCloudTopicId();
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);
  return {
    isLoading
  };
};
exports.useSyncUrlDetails = useSyncUrlDetails;