"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transformCustomScriptsToOptions = exports.createSelectionHandler = exports.createKeyDownHandler = exports.checkActionCancelPermission = void 0;
var _constants = require("../../../../../common/endpoint/service/response_actions/constants");
var _cancel_authz_utils = require("../../../../../common/endpoint/service/authz/cancel_authz_utils");
var _translations = require("../../../common/translations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Type representing a pending action item for cancellation
 */

/**
 * Generic handler for option selection in selectable components
 */
const createSelectionHandler = (onChange, state) => {
  return (_newOptions, _event, changedOption) => {
    if (changedOption.checked === 'on' && 'value' in changedOption) {
      const newState = {
        ...state,
        isPopoverOpen: false,
        selectedOption: changedOption.data
      };
      onChange({
        value: String(changedOption.value || ''),
        valueText: String(changedOption.value || ''),
        store: newState
      });
    } else {
      const newState = {
        ...state,
        isPopoverOpen: false,
        selectedOption: undefined
      };
      onChange({
        value: '',
        valueText: '',
        store: newState
      });
    }
  };
};

/**
 * Generic keyboard event handler for search input
 */
exports.createSelectionHandler = createSelectionHandler;
const createKeyDownHandler = event => {
  // Only stop propagation for typing keys, not for navigation keys - otherwise input lose focus
  if (!['Enter', 'ArrowUp', 'ArrowDown', 'Escape'].includes(event.key)) {
    event.stopPropagation();
  }
};

/**
 * Transform custom scripts data to selectable options
 */
exports.createKeyDownHandler = createKeyDownHandler;
const transformCustomScriptsToOptions = (data, selectedValue) => {
  return data.map(script => {
    const isChecked = script.name === selectedValue;
    return {
      label: script.name,
      value: script.name,
      description: script.description,
      checked: isChecked ? 'on' : undefined,
      data: script
    };
  });
};

/**
 * Check if user has permission to cancel a specific action
 */
exports.transformCustomScriptsToOptions = transformCustomScriptsToOptions;
const checkActionCancelPermission = (command, endpointPrivileges) => {
  const displayCommand = _constants.RESPONSE_ACTION_API_COMMAND_TO_CONSOLE_COMMAND_MAP[command] || command;
  try {
    const canCancel = (0, _cancel_authz_utils.canUserCancelCommand)(endpointPrivileges, command);
    if (!canCancel) {
      return {
        canCancel: false,
        reason: (0, _translations.getCancelPermissionDeniedMessage)(displayCommand)
      };
    }
    return {
      canCancel: true
    };
  } catch (error) {
    return {
      canCancel: false,
      reason: (0, _translations.getPermissionVerificationErrorMessage)(displayCommand)
    };
  }
};
exports.checkActionCancelPermission = checkActionCancelPermission;