"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseCommandInput = exports.getCommandNameFromTextInput = exports.getArgumentsForCommand = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const parseInputString = rawInput => {
  const input = rawInput.trim();
  const response = {
    name: getCommandNameFromTextInput(input),
    args: {}
  };
  if (!input) {
    return response;
  }
  const inputFirstSpacePosition = input.indexOf(' ');
  const rawArguments = inputFirstSpacePosition === -1 ? [] : input.substring(inputFirstSpacePosition).trim().split(/--/);
  for (const rawArg of rawArguments) {
    const argNameAndValueTrimmedString = rawArg.trim();
    if (argNameAndValueTrimmedString) {
      // rawArgument possible values here are:
      //    'option=something'
      //    'option'
      //    'option something
      // These all having possible spaces before and after

      const firstSpaceOrEqualSign = /[ =]/.exec(argNameAndValueTrimmedString);

      // Grab the argument name
      const argName = (firstSpaceOrEqualSign ? argNameAndValueTrimmedString.substring(0, firstSpaceOrEqualSign.index).trim() : argNameAndValueTrimmedString).trim();
      if (argName) {
        if (!response.args[argName]) {
          response.args[argName] = [];
        }

        // if this argument name has a value, then process that
        if (argName !== argNameAndValueTrimmedString && firstSpaceOrEqualSign) {
          let newArgValue = argNameAndValueTrimmedString.substring(firstSpaceOrEqualSign.index + 1).trim().replace(/\\-\\-/g, '--');
          if (newArgValue.charAt(0) === '"') {
            newArgValue = newArgValue.substring(1);
          }
          if (newArgValue.charAt(newArgValue.length - 1) === '"') {
            newArgValue = newArgValue.substring(0, newArgValue.length - 1);
          }
          response.args[argName].push(newArgValue);
        } else {
          // Argument has not value (bare), set it to empty string
          response.args[argName].push(true);
        }
      }
    }
  }
  return response;
};
class ParsedCommand {
  constructor(input) {
    (0, _defineProperty2.default)(this, "name", void 0);
    (0, _defineProperty2.default)(this, "args", void 0);
    (0, _defineProperty2.default)(this, "hasArgs", void 0);
    this.input = input;
    const parseInput = parseInputString(input);
    this.name = parseInput.name;
    this.args = parseInput.args;
    this.hasArgs = Object.keys(this.args).length > 0;
  }
  hasArg(argName) {
    return argName in this.args;
  }
}
const parseCommandInput = input => {
  return new ParsedCommand(input);
};
exports.parseCommandInput = parseCommandInput;
const getCommandNameFromTextInput = input => {
  const trimmedInput = input.trimStart();
  if (!trimmedInput) {
    return '';
  }
  const firstSpacePosition = input.indexOf(' ');
  if (firstSpacePosition === -1) {
    return trimmedInput;
  }
  return trimmedInput.substring(0, firstSpacePosition);
};
exports.getCommandNameFromTextInput = getCommandNameFromTextInput;
const getArgumentsForCommand = command => {
  let requiredArgs = '';
  let optionalArgs = '';
  const exclusiveOrArgs = [];
  if (command.args) {
    for (const [argName, argDefinition] of Object.entries(command.args)) {
      if (argDefinition.required) {
        if (requiredArgs.length) {
          requiredArgs += ' ';
        }
        requiredArgs += `--${argName}`;
      } else if (argDefinition.exclusiveOr) {
        exclusiveOrArgs.push(`--${argName}`);
      } else {
        if (optionalArgs.length) {
          optionalArgs += ' ';
        }
        optionalArgs += `--${argName}`;
      }
    }
  }
  const buildArgumentText = ({
    required,
    exclusive,
    optional
  }) => {
    return `${required ? required : ''}${exclusive ? ` ${exclusive}` : ''} ${optional && optional.length > 0 ? `[${optional}]` : ''}`.trim();
  };
  return exclusiveOrArgs.length > 0 ? exclusiveOrArgs.map(exclusiveArg => {
    return buildArgumentText({
      required: requiredArgs,
      exclusive: exclusiveArg,
      optional: optionalArgs
    });
  }) : requiredArgs || optionalArgs ? [buildArgumentText({
    required: requiredArgs,
    optional: optionalArgs
  })] : [];
};
exports.getArgumentsForCommand = getArgumentsForCommand;