"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.entriesToConditionEntriesMap = exports.entriesToConditionEntries = exports.conditionEntriesToEntries = void 0;
var _securitysolutionUtils = require("@kbn/securitysolution-utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const OPERATOR_VALUE = 'included';
const hashType = hash => {
  switch (hash.length) {
    case 32:
      return 'md5';
    case 40:
      return 'sha1';
    case 64:
      return 'sha256';
  }
};
const createEntryMatch = (field, value) => {
  return {
    field,
    value,
    type: 'match',
    operator: OPERATOR_VALUE
  };
};
const createEntryMatchAny = (field, value) => {
  return {
    field,
    value,
    type: 'match_any',
    operator: OPERATOR_VALUE
  };
};
const createEntryMatchWildcard = (field, value) => {
  return {
    field,
    value,
    type: 'wildcard',
    operator: OPERATOR_VALUE
  };
};
const createEntryNested = (field, entries) => {
  return {
    field,
    entries,
    type: 'nested'
  };
};
function groupHashEntry(prefix, conditionEntry) {
  const entriesArray = [];
  if (!Array.isArray(conditionEntry.value)) {
    const entry = createEntryMatch(`${prefix}${_securitysolutionUtils.EntryFieldType.HASH}${hashType(conditionEntry.value)}`, conditionEntry.value.toLowerCase());
    entriesArray.push(entry);
    return entriesArray;
  }
  const hashTypeGroups = conditionEntry.value.reduce((memo, val) => {
    const type = hashType(val);
    if (!type) return memo;
    return {
      ...memo,
      [type]: [...memo[type], val]
    };
  }, {
    md5: [],
    sha1: [],
    sha256: []
  });
  Object.entries(hashTypeGroups).forEach(([type, values]) => {
    if (!values.length) {
      return;
    }
    const entry = createEntryMatchAny(`${prefix}${_securitysolutionUtils.EntryFieldType.HASH}${type}`, values.map(value => value.toLowerCase()));
    entriesArray.push(entry);
  });
  return entriesArray;
}
function createNestedSignatureEntry(field, value, isTrustedApp = false) {
  const subjectNameMatch = Array.isArray(value) ? createEntryMatchAny('subject_name', value) : createEntryMatch('subject_name', value);
  const nestedEntries = [];
  if (isTrustedApp) nestedEntries.push(createEntryMatch('trusted', 'true'));
  nestedEntries.push(subjectNameMatch);
  return createEntryNested(field, nestedEntries);
}
function createWildcardPathEntry(field, value) {
  return Array.isArray(value) ? createEntryMatchAny(field, value) : createEntryMatchWildcard(field, value);
}
function createPathEntry(field, value) {
  return Array.isArray(value) ? createEntryMatchAny(field, value) : createEntryMatch(field, value);
}
const conditionEntriesToEntries = (conditionEntries, isTrustedApp = false) => {
  const entriesArray = [];
  conditionEntries.forEach(conditionEntry => {
    if (conditionEntry.field.includes(_securitysolutionUtils.EntryFieldType.HASH)) {
      const prefix = conditionEntry.field.split('.')[0];
      groupHashEntry(prefix, conditionEntry).forEach(entry => entriesArray.push(entry));
    } else if (conditionEntry.field.includes(_securitysolutionUtils.EntryFieldType.SIGNER)) {
      const entry = createNestedSignatureEntry(conditionEntry.field, conditionEntry.value, isTrustedApp);
      entriesArray.push(entry);
    } else if ((conditionEntry.field.includes(_securitysolutionUtils.EntryFieldType.EXECUTABLE) || conditionEntry.field.includes(_securitysolutionUtils.EntryFieldType.PATH)) && conditionEntry.type === 'wildcard') {
      const entry = createWildcardPathEntry(conditionEntry.field, conditionEntry.value);
      entriesArray.push(entry);
    } else {
      const entry = createPathEntry(conditionEntry.field, conditionEntry.value);
      entriesArray.push(entry);
    }
  });
  return entriesArray;
};
exports.conditionEntriesToEntries = conditionEntriesToEntries;
const createConditionEntry = (field, type, value) => {
  return {
    field,
    value,
    type,
    operator: OPERATOR_VALUE
  };
};
function createWildcardHashField(field) {
  const prefix = field.split('.')[0];
  return `${prefix}${_securitysolutionUtils.EntryFieldType.HASH}*`;
}
const entriesToConditionEntriesMap = entries => {
  return entries.reduce((memo, entry) => {
    const field = entry.field;
    if (field.includes(_securitysolutionUtils.EntryFieldType.HASH) && entry.type === 'match') {
      const wildcardHashField = createWildcardHashField(field);
      return {
        ...memo,
        [wildcardHashField]: createConditionEntry(wildcardHashField, entry.type, entry.value)
      };
    } else if (field.includes(_securitysolutionUtils.EntryFieldType.HASH) && entry.type === 'match_any') {
      var _ref, _memo$wildcardHashFie;
      const wildcardHashField = createWildcardHashField(field);
      const currentValues = (_ref = (_memo$wildcardHashFie = memo[wildcardHashField]) === null || _memo$wildcardHashFie === void 0 ? void 0 : _memo$wildcardHashFie.value) !== null && _ref !== void 0 ? _ref : [];
      return {
        ...memo,
        [wildcardHashField]: createConditionEntry(wildcardHashField, entry.type, [...currentValues, ...entry.value])
      };
    } else if ((field.includes(_securitysolutionUtils.EntryFieldType.EXECUTABLE) || field.includes(_securitysolutionUtils.EntryFieldType.PATH)) && (entry.type === 'match' || entry.type === 'match_any' || entry.type === 'wildcard')) {
      return {
        ...memo,
        [field]: createConditionEntry(field, entry.type, entry.value)
      };
    } else if (field.includes(_securitysolutionUtils.EntryFieldType.SIGNER) && entry.type === 'nested') {
      const subjectNameCondition = entry.entries.find(subEntry => {
        return subEntry.field === 'subject_name' && (subEntry.type === 'match' || subEntry.type === 'match_any');
      });
      if (subjectNameCondition) {
        return {
          ...memo,
          [field]: createConditionEntry(field, subjectNameCondition.type, subjectNameCondition.value)
        };
      }
    }
    return memo;
  }, {});
};
exports.entriesToConditionEntriesMap = entriesToConditionEntriesMap;
const entriesToConditionEntries = entries => {
  const conditionEntriesMap = entriesToConditionEntriesMap(entries);
  return Object.values(conditionEntriesMap).reduce((memo, entry) => {
    if (!entry) return memo;
    return [...memo, entry];
  }, []);
};
exports.entriesToConditionEntries = entriesToConditionEntries;