"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useSecurityJobs = void 0;
var _react = require("react");
var _constants = require("../../../../../common/constants");
var _has_ml_admin_permissions = require("../../../../../common/machine_learning/has_ml_admin_permissions");
var _has_ml_license = require("../../../../../common/machine_learning/has_ml_license");
var _use_app_toasts = require("../../../hooks/use_app_toasts");
var _kibana = require("../../../lib/kibana");
var _use_fetch_jobs_summary_query = require("../../ml/hooks/use_fetch_jobs_summary_query");
var _use_ml_capabilities = require("../../ml/hooks/use_ml_capabilities");
var i18n = _interopRequireWildcard(require("../../ml/translations"));
var _use_fetch_modules_query = require("./use_fetch_modules_query");
var _use_fetch_recognizer_query = require("./use_fetch_recognizer_query");
var _use_security_jobs_helpers = require("./use_security_jobs_helpers");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Compiles a collection of SecurityJobs, which are a list of all jobs relevant to the Security Solution App. This
 * includes all installed jobs in the `Security` ML group, and all jobs within ML Modules defined in
 * ml_module (whether installed or not). Use the corresponding helper functions to filter the job
 * list as necessary. E.g. installed jobs, running jobs, etc.
 *
 * NOTE: If the user is not an ml admin, jobs will be empty and isMlAdmin will be false.
 * If you only need installed jobs, try the {@link useInstalledSecurityJobs} hook.
 *
 */
const useSecurityJobs = () => {
  const mlCapabilities = (0, _use_ml_capabilities.useMlCapabilities)();
  const [securitySolutionDefaultIndex] = (0, _kibana.useUiSetting$)(_constants.DEFAULT_INDEX_KEY);
  const {
    addError
  } = (0, _use_app_toasts.useAppToasts)();
  const isMlAdmin = (0, _has_ml_admin_permissions.hasMlAdminPermissions)(mlCapabilities);
  const isLicensed = (0, _has_ml_license.hasMlLicense)(mlCapabilities);
  const isMlEnabled = isMlAdmin && isLicensed;
  const onError = (0, _react.useCallback)(error => {
    addError(error, {
      title: i18n.SIEM_JOB_FETCH_FAILURE
    });
  }, [addError]);
  const {
    data: jobSummaryData,
    isFetching: isJobSummaryFetching,
    refetch: refetchJobsSummary
  } = (0, _use_fetch_jobs_summary_query.useFetchJobsSummaryQuery)({}, {
    enabled: isMlEnabled,
    onError
  });
  const {
    data: modulesData,
    isFetching: isModulesFetching,
    refetch: refetchModules
  } = (0, _use_fetch_modules_query.useFetchModulesQuery)({}, {
    enabled: isMlEnabled,
    onError
  });
  const {
    data: compatibleModules,
    isFetching: isRecognizerFetching,
    refetch: refetchRecognizer
  } = (0, _use_fetch_recognizer_query.useFetchRecognizerQuery)({
    indexPatternName: securitySolutionDefaultIndex
  }, {
    enabled: isMlEnabled,
    onError
  });
  const refetch = (0, _react.useCallback)(() => {
    refetchJobsSummary();
    refetchModules();
    refetchRecognizer();
  }, [refetchJobsSummary, refetchModules, refetchRecognizer]);
  const jobs = (0, _react.useMemo)(() => {
    if (jobSummaryData && modulesData && compatibleModules) {
      return (0, _use_security_jobs_helpers.createSecurityJobs)(jobSummaryData, modulesData, compatibleModules);
    }
    return [];
  }, [compatibleModules, jobSummaryData, modulesData]);
  return {
    isLicensed,
    isMlAdmin,
    jobs,
    loading: isJobSummaryFetching || isModulesFetching || isRecognizerFetching,
    refetch
  };
};
exports.useSecurityJobs = useSecurityJobs;