"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseAssetCriticalityCsvRow = exports.isErrorResult = void 0;
var _i18n = require("@kbn/i18n");
var _constants = require("./constants");
var _types = require("../types");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_COLUMN_CHARS = 1000;
const validationErrorWithMessage = message => {
  return {
    valid: false,
    error: message
  };
};
const isErrorResult = result => {
  return !result.valid;
};
exports.isErrorResult = isErrorResult;
const trimColumn = column => {
  return column.length > MAX_COLUMN_CHARS ? `${column.substring(0, MAX_COLUMN_CHARS)}...` : column;
};
const parseAssetCriticalityCsvRow = row => {
  if (row.length !== 3) {
    return validationErrorWithMessage(_i18n.i18n.translate('xpack.securitySolution.assetCriticality.csvUpload.expectedColumnsError', {
      defaultMessage: 'Expected 3 columns, got {rowLength}',
      values: {
        rowLength: row.length
      }
    }));
  }
  const [entityType, idValue, criticalityLevel] = row;
  if (!entityType) {
    return validationErrorWithMessage(_i18n.i18n.translate('xpack.securitySolution.assetCriticality.csvUpload.missingEntityTypeError', {
      defaultMessage: 'Missing entity type'
    }));
  }
  if (!idValue) {
    return validationErrorWithMessage(_i18n.i18n.translate('xpack.securitySolution.assetCriticality.csvUpload.missingIdError', {
      defaultMessage: 'Missing identifier'
    }));
  }
  if (idValue.length > MAX_COLUMN_CHARS) {
    return validationErrorWithMessage(_i18n.i18n.translate('xpack.securitySolution.assetCriticality.csvUpload.idTooLongError', {
      defaultMessage: 'Identifier is too long, expected less than {maxChars} characters, got {idLength}',
      values: {
        maxChars: MAX_COLUMN_CHARS,
        idLength: idValue.length
      }
    }));
  }
  if (!criticalityLevel) {
    return validationErrorWithMessage(_i18n.i18n.translate('xpack.securitySolution.assetCriticality.csvUpload.missingCriticalityError', {
      defaultMessage: 'Missing criticality level'
    }));
  }
  const lowerCaseCriticalityLevel = criticalityLevel.toLowerCase();
  if (!isValidCriticalityLevel(lowerCaseCriticalityLevel)) {
    return validationErrorWithMessage(_i18n.i18n.translate('xpack.securitySolution.assetCriticality.csvUpload.invalidCriticalityError', {
      defaultMessage: 'Invalid criticality level "{criticalityLevel}", expected one of {validLevels}',
      values: {
        criticalityLevel: trimColumn(criticalityLevel),
        validLevels: _constants.ValidCriticalityLevels.join(', ')
      }
    }));
  }
  const enabledEntityTypes = (0, _utils.getEntityAnalyticsEntityTypes)();
  if (!enabledEntityTypes.includes(entityType)) {
    return validationErrorWithMessage(_i18n.i18n.translate('xpack.securitySolution.assetCriticality.csvUpload.invalidEntityTypeError', {
      defaultMessage: 'Invalid entity type "{entityType}", expected to be one of: {validTypes}',
      values: {
        entityType: trimColumn(entityType),
        validTypes: enabledEntityTypes.join(', ')
      }
    }));
  }
  const idField = _types.EntityTypeToIdentifierField[entityType];
  return {
    valid: true,
    record: {
      idField,
      idValue,
      criticalityLevel: lowerCaseCriticalityLevel
    }
  };
};
exports.parseAssetCriticalityCsvRow = parseAssetCriticalityCsvRow;
const isValidCriticalityLevel = criticalityLevel => {
  return _constants.ValidCriticalityLevels.includes(criticalityLevel);
};