"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isCancelFeatureAvailable = exports.checkCancelPermission = exports.canUserCancelCommand = void 0;
var _constants = require("../response_actions/constants");
var _is_response_action_supported = require("../response_actions/is_response_action_supported");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Checks if cancel operations are available for the given agent type and environment.
 * This is a general capability check, not command-specific.
 *
 * Uses `canCancelAction` as the base permission because cancel operations require
 * meaningful response action capabilities (not just read access). This permission is calculated
 * based on having least one non-release/non-cancel response action permission.
 *
 * @param authz - The user's endpoint authorization permissions
 * @param featureFlags - Experimental features configuration
 * @param agentType - The agent type (endpoint, sentinel_one, crowdstrike, microsoft_defender_endpoint)
 * @returns true if cancel feature is available for the agent type
 */
const isCancelFeatureAvailable = (authz, featureFlags, agentType) => {
  // Check base access to security solution
  if (!authz.canCancelAction) {
    return false;
  }

  // Check agent type
  if (agentType !== 'microsoft_defender_endpoint') {
    return false;
  }

  // Check if Microsoft Defender Endpoint cancel feature is enabled
  if (!featureFlags.microsoftDefenderEndpointCancelEnabled) {
    return false;
  }

  // Check if agent type supports cancel operations
  return (0, _is_response_action_supported.isActionSupportedByAgentType)(agentType, 'cancel', 'manual');
};

/**
 * Checks if user can cancel a specific command.
 * Assumes cancel feature is available (checked separately).
 *
 * @param authz - The user's endpoint authorization permissions
 * @param command - The response action command being cancelled
 * @returns true if the user has permission to cancel the specified command
 */
exports.isCancelFeatureAvailable = isCancelFeatureAvailable;
const canUserCancelCommand = (authz, command) => {
  const consoleCommand = _constants.RESPONSE_ACTION_API_COMMAND_TO_CONSOLE_COMMAND_MAP[command];
  const requiredAuthzKey = _constants.RESPONSE_CONSOLE_ACTION_COMMANDS_TO_REQUIRED_AUTHZ[consoleCommand];
  return authz[requiredAuthzKey];
};

/**
 * Complete check for cancel permission
 * Combines feature availability and command-specific permission.
 *
 * @param authz - The user's endpoint authorization permissions
 * @param featureFlags - Experimental features configuration
 * @param agentType - The agent type
 * @param command - The response action command being cancelled
 * @returns true if the user has permission to cancel the command
 */
exports.canUserCancelCommand = canUserCancelCommand;
const checkCancelPermission = (authz, featureFlags, agentType, command) => {
  return isCancelFeatureAvailable(authz, featureFlags, agentType) && canUserCancelCommand(authz, command);
};
exports.checkCancelPermission = checkCancelPermission;