"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.indexFleetServerAgent = exports.indexFleetAgentForHost = exports.deleteIndexedFleetAgents = exports.buildFleetAgentBulkCreateOperations = void 0;
var _common = require("@kbn/fleet-plugin/common");
var _common2 = require("@kbn/spaces-plugin/common");
var _usage_tracker = require("./usage_tracker");
var _fleet_agent_generator = require("../data_generators/fleet_agent_generator");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const defaultFleetAgentGenerator = new _fleet_agent_generator.FleetAgentGenerator();
/**
 * Indexes a Fleet Agent
 * (NOTE: ensure that fleet is setup first before calling this loading function)
 *
 * @param esClient
 * @param endpointHost
 * @param agentPolicyId
 * @param [kibanaVersion]
 * @param [fleetAgentGenerator]
 */
const indexFleetAgentForHost = exports.indexFleetAgentForHost = _usage_tracker.usageTracker.track('indexFleetAgentForHost', async (esClient, endpointHost, agentPolicyId, kibanaVersion = '8.0.0', fleetAgentGenerator = defaultFleetAgentGenerator, spaceId = _common2.DEFAULT_SPACE_ID) => {
  const agentDoc = generateFleetAgentEsHitForEndpointHost(endpointHost, agentPolicyId, kibanaVersion, fleetAgentGenerator, spaceId);
  await esClient.index({
    index: agentDoc._index,
    id: agentDoc._id,
    body: agentDoc._source,
    op_type: 'create',
    refresh: 'wait_for'
  }).catch(_utils.wrapErrorAndRejectPromise);
  return {
    fleetAgentsIndex: agentDoc._index,
    // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
    agents: [agentDoc._source]
  };
});
const generateFleetAgentEsHitForEndpointHost = (endpointHost, agentPolicyId, kibanaVersion = '8.0.0', fleetAgentGenerator = defaultFleetAgentGenerator, spaceId = _common2.DEFAULT_SPACE_ID) => {
  const esHit = fleetAgentGenerator.generateEsHit({
    _id: endpointHost.agent.id,
    _source: {
      agent: {
        id: endpointHost.agent.id,
        version: endpointHost.agent.version
      },
      local_metadata: {
        elastic: {
          agent: {
            id: endpointHost.agent.id,
            version: kibanaVersion
          }
        },
        host: {
          ...endpointHost.host
        },
        os: {
          ...endpointHost.host.os
        }
      },
      policy_id: agentPolicyId,
      namespaces: Array.isArray(spaceId) ? spaceId : [spaceId]
    }
  });

  // Set the agent status to Healthy
  // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
  esHit._source.components = [{
    id: 'endpoint-0',
    type: 'endpoint',
    status: 'HEALTHY',
    message: 'Running as external service',
    units: [{
      id: 'endpoint-1',
      type: 'input',
      status: 'HEALTHY',
      message: 'Protecting machine'
    }, {
      id: 'shipper',
      type: 'output',
      status: 'HEALTHY',
      message: 'Connected over GRPC'
    }]
  }];
  return esHit;
};
/**
 * Creates an array of ES records with Fleet Agents that are associated with the provided set of Endpoint Agents.
 * Array can be used with the `bulk()` API's `operations` option.
 * @param endpoints
 * @param agentPolicyId
 * @param kibanaVersion
 * @param fleetAgentGenerator
 */
const buildFleetAgentBulkCreateOperations = ({
  endpoints,
  agentPolicyId,
  kibanaVersion = '8.0.0',
  fleetAgentGenerator = defaultFleetAgentGenerator,
  spaceId = _common2.DEFAULT_SPACE_ID
}) => {
  const response = {
    operations: [],
    agents: [],
    fleetAgentsIndex: _common.AGENTS_INDEX
  };
  for (const endpointHost of endpoints) {
    const agentDoc = generateFleetAgentEsHitForEndpointHost(endpointHost, agentPolicyId, kibanaVersion, fleetAgentGenerator, spaceId);
    response.operations.push({
      create: {
        _index: agentDoc._index,
        _id: agentDoc._id
      }
    }, agentDoc._source);
    // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
    response.agents.push(agentDoc._source);
  }
  return response;
};
exports.buildFleetAgentBulkCreateOperations = buildFleetAgentBulkCreateOperations;
const deleteIndexedFleetAgents = async (esClient, indexedData) => {
  const response = {
    agents: undefined
  };
  if (indexedData.agents.length) {
    response.agents = await esClient.deleteByQuery({
      index: `${indexedData.fleetAgentsIndex}-*`,
      wait_for_completion: true,
      query: {
        bool: {
          filter: [{
            terms: {
              'local_metadata.elastic.agent.id': indexedData.agents.map(agent => agent.local_metadata.elastic.agent.id)
            }
          }]
        }
      }
    }).catch(_utils.wrapErrorAndRejectPromise);
  }
  return response;
};
exports.deleteIndexedFleetAgents = deleteIndexedFleetAgents;
const indexFleetServerAgent = async (esClient, log = (0, _utils.createToolingLogger)(), overrides = {}) => {
  const doc = defaultFleetAgentGenerator.generateEsHit({
    _source: overrides
  });
  const indexRequest = {
    index: doc._index,
    id: doc._id,
    body: doc._source,
    op_type: 'create',
    refresh: 'wait_for'
  };
  log.verbose(`Indexing new fleet agent with:\n${JSON.stringify(indexRequest, null, 2)}`);
  await esClient.index(indexRequest).catch(_utils.wrapErrorAndRejectPromise);
  return {
    fleetAgentsIndex: doc._index,
    // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
    agents: [doc._source]
  };
};
exports.indexFleetServerAgent = indexFleetServerAgent;