"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.indexEndpointHeartbeats = exports.deleteIndexedEndpointHeartbeats = void 0;
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const indexEndpointHeartbeats = async (esClient, log, count = 1, unbilledCount = 1) => {
  const startTime = new Date();
  log.debug(`Indexing ${count} endpoint heartbeats with startTime = ${startTime}`);
  const docs = Array.from({
    length: count
  }).map((_, i) => {
    const ingested = new Date(startTime.getTime() + i).toISOString();
    const heartbeatDoc = {
      '@timestamp': '2024-06-11T13:03:37Z',
      agent: {
        id: `agent-${i}`
      },
      event: {
        agent_id_status: 'auth_metadata_missing',
        ingested
      }
    };
    // billable: true and missing billable are billed
    if (i % 2) {
      heartbeatDoc.billable = true;
    }
    return heartbeatDoc;
  });
  const unbilledDocs = Array.from({
    length: unbilledCount
  }).map((_, i) => {
    const ingested = new Date(startTime.getTime() + i).toISOString();
    return {
      '@timestamp': '2024-06-11T13:03:37Z',
      agent: {
        id: `agent-billable-false-${i}`
      },
      event: {
        agent_id_status: 'auth_metadata_missing',
        ingested
      },
      billable: false
    };
  });
  const operations = docs.concat(unbilledDocs).flatMap(doc => [{
    index: {
      // simulating different namespaces
      _index: `.logs-endpoint.heartbeat-${doc.agent.id.slice(-1)}`,
      op_type: 'create'
    }
  }, doc]);
  const response = await esClient.bulk({
    refresh: 'wait_for',
    operations
  });
  if (response.errors) {
    log.error(`There was an error indexing endpoint heartbeats ${JSON.stringify(response.items, null, 2)}`);
  } else {
    log.debug(`Indexed ${docs.length} endpoint heartbeats successfully`);
  }
  return {
    data: response,
    cleanup: deleteIndexedEndpointHeartbeats.bind(null, esClient, response, log)
  };
};
exports.indexEndpointHeartbeats = indexEndpointHeartbeats;
const deleteIndexedEndpointHeartbeats = async (esClient, indexedHeartbeats, log = (0, _utils.createToolingLogger)()) => {
  let response = {
    took: 0,
    errors: false,
    items: []
  };
  if (indexedHeartbeats.items.length) {
    const idsToDelete = indexedHeartbeats.items.filter(item => item.create).map(item => {
      var _item$create, _item$create2;
      return {
        delete: {
          _index: (_item$create = item.create) === null || _item$create === void 0 ? void 0 : _item$create._index,
          _id: (_item$create2 = item.create) === null || _item$create2 === void 0 ? void 0 : _item$create2._id
        }
      };
    });
    if (idsToDelete.length) {
      response = await esClient.bulk({
        operations: idsToDelete
      });
      log.debug('Indexed endpoint heartbeats deleted successfully');
    }
  }
  return {
    data: response
  };
};
exports.deleteIndexedEndpointHeartbeats = deleteIndexedEndpointHeartbeats;