"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.definePutRolesRoutes = definePutRolesRoutes;
var _configSchema = require("@kbn/config-schema");
var _coreSecurityServer = require("@kbn/core-security-server");
var _lib = require("./lib");
var _model = require("./model");
var _constants = require("../../../../common/constants");
var _errors = require("../../../errors");
var _lib2 = require("../../../lib");
var _licensed_route_handler = require("../../licensed_route_handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function definePutRolesRoutes({
  router,
  authz,
  getFeatures,
  getFeatureUsageService
}) {
  router.versioned.put({
    path: '/api/security/role/{name}',
    access: 'public',
    summary: `Create or update a role`,
    description: 'Create a new Kibana role or update the attributes of an existing role. Kibana roles are stored in the Elasticsearch native realm.',
    options: {
      tags: ['oas-tag:roles']
    },
    security: {
      authz: _coreSecurityServer.AuthzDisabled.delegateToESClient
    }
  }).addVersion({
    version: _constants.API_VERSIONS.roles.public.v1,
    validate: {
      request: {
        params: _configSchema.schema.object({
          name: _configSchema.schema.string({
            minLength: 1,
            maxLength: 1024,
            meta: {
              description: 'The role name.'
            }
          })
        }),
        query: _configSchema.schema.object({
          createOnly: _configSchema.schema.boolean({
            defaultValue: false,
            meta: {
              description: 'When true, a role is not overwritten if it already exists.'
            }
          })
        }),
        body: (0, _model.getPutPayloadSchema)(() => {
          const privileges = authz.privileges.get();
          return {
            global: Object.keys(privileges.global),
            space: Object.keys(privileges.space)
          };
        })
      },
      response: {
        204: {
          description: 'Indicates a successful call.'
        }
      }
    }
  }, (0, _licensed_route_handler.createLicensedRouteHandler)(async (context, request, response) => {
    const {
      name
    } = request.params;
    const {
      createOnly
    } = request.query;
    try {
      const esClient = (await context.core).elasticsearch.client;
      const [features, rawRoles] = await Promise.all([getFeatures(), esClient.asCurrentUser.security.getRole({
        name: request.params.name
      }, {
        ignore: [404]
      })]);
      const {
        validationErrors
      } = (0, _lib2.validateKibanaPrivileges)(features, request.body.kibana);
      if (validationErrors.length) {
        return response.badRequest({
          body: {
            message: `Role cannot be updated due to validation errors: ${JSON.stringify(validationErrors)}`
          }
        });
      }
      if (createOnly && !!rawRoles[name]) {
        return response.conflict({
          body: {
            message: `Role already exists and cannot be created: ${name}`
          }
        });
      }
      const body = (0, _model.transformPutPayloadToElasticsearchRole)(request.body, authz.applicationName, rawRoles[name] ? rawRoles[name].applications : []);
      await esClient.asCurrentUser.security.putRole({
        name: request.params.name,
        ...body
      });
      if ((0, _lib.roleGrantsSubFeaturePrivileges)(features, request.body)) {
        getFeatureUsageService().recordSubFeaturePrivilegeUsage();
      }
      return response.noContent();
    } catch (error) {
      return response.customError((0, _errors.wrapIntoCustomErrorResponse)(error));
    }
  }));
}