"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initAPIAuthorization = initAPIAuthorization;
var _server = require("@kbn/core/server");
var _coreSecurityServer = require("@kbn/core-security-server");
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isAuthzDisabled = authz => {
  return (authz === null || authz === void 0 ? void 0 : authz.enabled) === false;
};
const isReservedPrivilegeSet = privilege => {
  return Object.hasOwn(_server.ReservedPrivilegesSet, privilege);
};
function initAPIAuthorization(http, {
  actions,
  checkPrivilegesDynamicallyWithRequest,
  checkPrivilegesWithRequest,
  mode,
  getCurrentUser,
  getSecurityConfig
}, logger) {
  http.registerOnPostAuth(async (request, response, toolkit) => {
    // if we aren't using RBAC for this request, just continue
    if (!mode.useRbacForRequest(request)) {
      return toolkit.next();
    }
    const security = request.route.options.security;
    if (isAuthzDisabled(security.authz)) {
      return toolkit.next();
    }
    const authz = security.authz;
    const normalizeRequiredPrivileges = async privileges => {
      const hasOperatorPrivileges = privileges.some(privilege => {
        var _privilege$allRequire;
        return privilege === _server.ReservedPrivilegesSet.operator || typeof privilege === 'object' && ((_privilege$allRequire = privilege.allRequired) === null || _privilege$allRequire === void 0 ? void 0 : _privilege$allRequire.includes(_server.ReservedPrivilegesSet.operator));
      });

      // nothing to normalize
      if (!hasOperatorPrivileges) {
        return privileges;
      }
      const securityConfig = await getSecurityConfig();

      // nothing to normalize
      if (securityConfig.operator_privileges.enabled) {
        return privileges;
      }
      return privileges.reduce((acc, privilege) => {
        if (typeof privilege === 'object') {
          var _privilege$allRequire2, _privilege$allRequire3, _privilege$allRequire4;
          const operatorPrivilegeIndex = (_privilege$allRequire2 = (_privilege$allRequire3 = privilege.allRequired) === null || _privilege$allRequire3 === void 0 ? void 0 : _privilege$allRequire3.findIndex(p => p === _server.ReservedPrivilegesSet.operator)) !== null && _privilege$allRequire2 !== void 0 ? _privilege$allRequire2 : -1;
          acc.push(operatorPrivilegeIndex !== -1 ? {
            ...privilege,
            // @ts-ignore wrong types for `toSpliced`
            allRequired: (_privilege$allRequire4 = privilege.allRequired) === null || _privilege$allRequire4 === void 0 ? void 0 : _privilege$allRequire4.toSpliced(operatorPrivilegeIndex, 1)
          } : privilege);
        } else if (privilege !== _server.ReservedPrivilegesSet.operator) {
          acc.push(privilege);
        }
        return acc;
      }, []);
    };

    // We need to normalize privileges to drop unintended privilege checks.
    // Operator privileges check should be only performed if the `operator_privileges` are enabled in config.
    const requiredPrivileges = await normalizeRequiredPrivileges(authz.requiredPrivileges);
    const {
      requestedPrivileges,
      requestedReservedPrivileges
    } = requiredPrivileges.reduce((acc, privilegeEntry) => {
      var _privilegeEntry$allRe, _privilegeEntry$anyRe;
      const privileges = typeof privilegeEntry === 'object' ? [...(0, _coreSecurityServer.unwindNestedSecurityPrivileges)((_privilegeEntry$allRe = privilegeEntry.allRequired) !== null && _privilegeEntry$allRe !== void 0 ? _privilegeEntry$allRe : []), ...(0, _coreSecurityServer.unwindNestedSecurityPrivileges)((_privilegeEntry$anyRe = privilegeEntry.anyRequired) !== null && _privilegeEntry$anyRe !== void 0 ? _privilegeEntry$anyRe : [])] : [privilegeEntry];
      for (const privilege of privileges) {
        if (isReservedPrivilegeSet(privilege)) {
          acc.requestedReservedPrivileges.push(privilege);
        } else {
          acc.requestedPrivileges.push(privilege);
        }
      }
      return acc;
    }, {
      requestedPrivileges: [],
      requestedReservedPrivileges: []
    });
    const checkPrivileges = checkPrivilegesDynamicallyWithRequest(request);
    const privilegeToApiOperation = privilege => privilege.replace(_constants.API_OPERATION_PREFIX, '');
    const kibanaPrivileges = {};
    if (requestedPrivileges.length > 0) {
      const checkPrivilegesResponse = await checkPrivileges({
        kibana: requestedPrivileges.map(permission => actions.api.get(permission))
      });
      for (const kbPrivilege of checkPrivilegesResponse.privileges.kibana) {
        kibanaPrivileges[privilegeToApiOperation(kbPrivilege.privilege)] = kbPrivilege.authorized;
      }
    }
    for (const reservedPrivilege of requestedReservedPrivileges) {
      if (reservedPrivilege === _server.ReservedPrivilegesSet.superuser) {
        const checkSuperuserPrivilegesResponse = await checkPrivilegesWithRequest(request).globally(_constants.SUPERUSER_PRIVILEGES);
        kibanaPrivileges[_server.ReservedPrivilegesSet.superuser] = checkSuperuserPrivilegesResponse.hasAllRequested;
      }
      if (reservedPrivilege === _server.ReservedPrivilegesSet.operator) {
        var _currentUser$operator;
        const currentUser = getCurrentUser(request);
        kibanaPrivileges[_server.ReservedPrivilegesSet.operator] = (_currentUser$operator = currentUser === null || currentUser === void 0 ? void 0 : currentUser.operator) !== null && _currentUser$operator !== void 0 ? _currentUser$operator : false;
      }
    }
    const hasRequestedPrivilege = kbPrivilege => {
      if (typeof kbPrivilege === 'object') {
        var _kbPrivilege$allRequi, _kbPrivilege$anyRequi;
        const allRequired = (_kbPrivilege$allRequi = kbPrivilege.allRequired) !== null && _kbPrivilege$allRequi !== void 0 ? _kbPrivilege$allRequi : [];
        const anyRequired = (_kbPrivilege$anyRequi = kbPrivilege.anyRequired) !== null && _kbPrivilege$anyRequi !== void 0 ? _kbPrivilege$anyRequi : [];
        return allRequired.every(privilege => typeof privilege === 'string' ? kibanaPrivileges[privilege] :
        // checking composite privileges
        privilege.anyOf.some(anyPrivilegeEntry => kibanaPrivileges[anyPrivilegeEntry])) && (!anyRequired.length || anyRequired.some(privilege => typeof privilege === 'string' ? kibanaPrivileges[privilege] :
        // checking composite privileges
        privilege.allOf.every(allPrivilegeEntry => kibanaPrivileges[allPrivilegeEntry])));
      }
      return kibanaPrivileges[kbPrivilege];
    };
    for (const privilege of requiredPrivileges) {
      if (!hasRequestedPrivilege(privilege)) {
        const missingPrivileges = Object.keys(kibanaPrivileges).filter(key => !kibanaPrivileges[key]);
        const forbiddenMessage = `API [${request.route.method.toUpperCase()} ${request.url.pathname}${request.url.search}] is unauthorized for user, this action is granted by the Kibana privileges [${missingPrivileges}]`;
        logger.warn(forbiddenMessage);
        return response.forbidden({
          body: {
            message: forbiddenMessage
          }
        });
      }
    }
    return toolkit.authzResultNext(kibanaPrivileges);
  });
}