"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RoleValidator = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _i18n = require("@kbn/i18n");
var _constants = require("../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class RoleValidator {
  constructor(options = {}) {
    (0, _defineProperty2.default)(this, "shouldValidate", void 0);
    (0, _defineProperty2.default)(this, "buildFlavor", void 0);
    this.shouldValidate = options.shouldValidate;
    this.buildFlavor = options.buildFlavor;
  }
  enableValidation() {
    this.shouldValidate = true;
  }
  disableValidation() {
    this.shouldValidate = false;
  }
  validateRoleName(role) {
    if (!this.shouldValidate) {
      return valid();
    }
    if (!role.name) {
      return invalid(_i18n.i18n.translate('xpack.security.management.editRole.validateRole.provideRoleNameWarningMessage', {
        defaultMessage: 'Please provide a role name.'
      }));
    }
    if (role.name.length > _constants.MAX_NAME_LENGTH) {
      return invalid(_i18n.i18n.translate('xpack.security.management.editRole.validateRole.nameLengthWarningMessage', {
        defaultMessage: 'Name must not exceed {maxLength} characters.',
        values: {
          maxLength: _constants.MAX_NAME_LENGTH
        }
      }));
    }
    if (role.name.trim() !== role.name) {
      return invalid(_i18n.i18n.translate('xpack.security.management.editRole.validateRole.nameWhitespaceWarningMessage', {
        defaultMessage: `Name must not contain leading or trailing spaces.`
      }));
    }
    if (this.buildFlavor === 'serverless' && !role.name.match(_constants.SERVERLESS_NAME_REGEX)) {
      return invalid(_i18n.i18n.translate('xpack.security.management.editRole.validateRole.serverlessNameAllowedCharactersWarningMessage', {
        defaultMessage: 'Name must contain only alphanumeric characters, and non-leading dots, hyphens, or underscores.'
      }));
    } else if (this.buildFlavor !== 'serverless' && !role.name.match(_constants.NAME_REGEX)) {
      return invalid(_i18n.i18n.translate('xpack.security.management.editRole.validateRole.nameAllowedCharactersWarningMessage', {
        defaultMessage: 'Name must contain only letters, numbers, spaces, punctuation and printable symbols.'
      }));
    }
    return valid();
  }
  validateRemoteClusterPrivileges(role) {
    var _role$elasticsearch$r;
    if (!this.shouldValidate) {
      return valid();
    }
    const areRemoteClustersInvalid = (_role$elasticsearch$r = role.elasticsearch.remote_cluster) === null || _role$elasticsearch$r === void 0 ? void 0 : _role$elasticsearch$r.some(remoteClusterPrivilege => {
      return this.validateRemoteClusterPrivilegeClusterField(remoteClusterPrivilege).isInvalid || this.validateRemoteClusterPrivilegePrivilegesField(remoteClusterPrivilege).isInvalid;
    });
    if (areRemoteClustersInvalid) {
      return invalid();
    }
    return valid();
  }
  validateIndexPrivileges(role) {
    if (!this.shouldValidate) {
      return valid();
    }
    if (!Array.isArray(role.elasticsearch.indices)) {
      throw new TypeError(_i18n.i18n.translate('xpack.security.management.editRole.validateRole.indicesTypeErrorMessage', {
        defaultMessage: 'Expected {elasticIndices} to be an array',
        values: {
          elasticIndices: '"role.elasticsearch.indices"'
        }
      }));
    }
    const areIndicesInvalid = role.elasticsearch.indices.reduce((isInvalid, indexPriv) => {
      if (this.validateIndexPrivilegeNamesField(indexPriv).isInvalid || this.validateIndexPrivilegePrivilegesField(indexPriv).isInvalid) {
        return true;
      }
      return isInvalid;
    }, false);
    if (areIndicesInvalid) {
      return invalid();
    }
    return valid();
  }
  validateRemoteIndexPrivileges(role) {
    if (!this.shouldValidate) {
      return valid();
    }
    if (!role.elasticsearch.remote_indices) {
      return valid();
    }
    if (!Array.isArray(role.elasticsearch.remote_indices)) {
      throw new TypeError(_i18n.i18n.translate('xpack.security.management.editRole.validateRole.indicesTypeErrorMessage', {
        defaultMessage: 'Expected {elasticIndices} to be an array',
        values: {
          elasticIndices: '"role.elasticsearch.remote_indices"'
        }
      }));
    }
    const areRemoteIndicesInvalid = role.elasticsearch.remote_indices.reduce((isInvalid, indexPriv) => {
      if (this.validateRemoteIndexPrivilegeClustersField(indexPriv).isInvalid || this.validateIndexPrivilegeNamesField(indexPriv).isInvalid || this.validateIndexPrivilegePrivilegesField(indexPriv).isInvalid) {
        return true;
      }
      return isInvalid;
    }, false);
    if (areRemoteIndicesInvalid) {
      return invalid();
    }
    return valid();
  }
  validateRemoteIndexPrivilegeClustersField(indexPrivilege) {
    if (!this.shouldValidate) {
      return valid();
    }

    // Ignore if all other fields are empty
    if (!indexPrivilege.names.length && !indexPrivilege.privileges.length) {
      return valid();
    }
    if (!indexPrivilege.clusters || !indexPrivilege.clusters.length) {
      return invalid(_i18n.i18n.translate('xpack.security.management.editRole.validateRole.oneRemoteClusterRequiredWarningMessage', {
        defaultMessage: 'Enter or select at least one remote cluster'
      }));
    }
    return valid();
  }
  validateIndexPrivilegeNamesField(indexPrivilege) {
    if (!this.shouldValidate) {
      return valid();
    }

    // Ignore if all other fields are empty
    if ((!('clusters' in indexPrivilege) || !indexPrivilege.clusters.length) && !indexPrivilege.privileges.length) {
      return valid();
    }
    if (!indexPrivilege.names.length) {
      return invalid(_i18n.i18n.translate('xpack.security.management.editRole.validateRole.oneIndexRequiredWarningMessage', {
        defaultMessage: 'Enter or select at least one index pattern'
      }));
    }
    return valid();
  }
  validateIndexPrivilegePrivilegesField(indexPrivilege) {
    if (!this.shouldValidate) {
      return valid();
    }

    // Ignore if all other fields are empty
    if ((!('clusters' in indexPrivilege) || !indexPrivilege.clusters.length) && !indexPrivilege.names.length) {
      return valid();
    }
    if (!indexPrivilege.privileges.length) {
      return invalid(_i18n.i18n.translate('xpack.security.management.editRole.validateRole.onePrivilegeRequiredWarningMessage', {
        defaultMessage: 'Enter or select at least one action'
      }));
    }
    return valid();
  }
  validateRemoteClusterPrivilegeClusterField(remoteClusterPrivilege) {
    if (!this.shouldValidate) {
      return valid();
    }

    // Ignore if all other fields are empty
    if (!remoteClusterPrivilege.privileges.length) {
      return valid();
    }
    if (!remoteClusterPrivilege.clusters.length) {
      return invalid(_i18n.i18n.translate('xpack.security.management.editRole.validateRole.oneClusterRequiredWarningMessage', {
        defaultMessage: 'Enter or select at least one cluster'
      }));
    }
    return valid();
  }
  validateRemoteClusterPrivilegePrivilegesField(remoteClusterPrivilege) {
    if (!this.shouldValidate) {
      return valid();
    }

    // Ignore if all other fields are empty
    if (!remoteClusterPrivilege.clusters.length) {
      return valid();
    }
    if (!remoteClusterPrivilege.privileges.length) {
      return invalid(_i18n.i18n.translate('xpack.security.management.editRole.validateRole.oneRemoteClusterPrivilegeRequiredWarningMessage', {
        defaultMessage: 'Enter or select at least one privilege'
      }));
    }
    return valid();
  }
  validateSelectedSpaces(spaceIds, privilege) {
    if (!this.shouldValidate) {
      return valid();
    }

    // If no assigned privilege, then no spaces are OK
    if (!privilege) {
      return valid();
    }
    if (Array.isArray(spaceIds) && spaceIds.length > 0) {
      return valid();
    }
    return invalid(_i18n.i18n.translate('xpack.security.management.editRole.validateRole.oneSpaceRequiredWarningMessage', {
      defaultMessage: 'At least one space is required'
    }));
  }
  validateSelectedPrivilege(spaceIds, privilege) {
    if (!this.shouldValidate) {
      return valid();
    }

    // If no assigned spaces, then a missing privilege is OK
    if (!spaceIds || spaceIds.length === 0) {
      return valid();
    }
    if (privilege) {
      return valid();
    }
    return invalid(_i18n.i18n.translate('xpack.security.management.editRole.validateRole.privilegeRequiredWarningMessage', {
      defaultMessage: 'Privilege is required'
    }));
  }
  validateSpacePrivileges(role) {
    if (!this.shouldValidate) {
      return valid();
    }
    const privileges = role.kibana || [];
    const arePrivilegesValid = privileges.every(assignedPrivilege => {
      return assignedPrivilege.base.length > 0 || Object.keys(assignedPrivilege.feature).length > 0;
    });
    if (arePrivilegesValid) {
      return valid();
    }
    return invalid();
  }
  validateForSave(role) {
    const {
      isInvalid: isNameInvalid
    } = this.validateRoleName(role);
    const {
      isInvalid: areIndicesInvalid
    } = this.validateIndexPrivileges(role);
    const {
      isInvalid: areRemoteIndicesInvalid
    } = this.validateRemoteIndexPrivileges(role);
    const {
      isInvalid: areSpacePrivilegesInvalid
    } = this.validateSpacePrivileges(role);
    const {
      isInvalid: areRemoteClusterPrivilegesInvalid
    } = this.validateRemoteClusterPrivileges(role);
    if (isNameInvalid || areIndicesInvalid || areRemoteIndicesInvalid || areSpacePrivilegesInvalid || areRemoteClusterPrivilegesInvalid) {
      return invalid();
    }
    return valid();
  }
}
exports.RoleValidator = RoleValidator;
function invalid(error) {
  return {
    isInvalid: true,
    error
  };
}
function valid() {
  return {
    isInvalid: false
  };
}