"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseFieldsCapabilities = exports.getModelIdFields = exports.fetchFields = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const EMBEDDING_TYPE = {
  sparse_embedding: 'sparse_vector',
  text_embedding: 'dense_vector'
};
const getModelIdFields = fieldCapsResponse => {
  const {
    fields
  } = fieldCapsResponse;
  return Object.keys(fields).reduce((acc, fieldKey) => {
    if (fieldKey.endsWith('model_id')) {
      const multiField = Object.keys(fields).filter(key => key.startsWith(fieldKey)).find(key => fields[key].keyword && fields[key].keyword.aggregatable);
      if (multiField) {
        acc.push({
          path: fieldKey,
          aggField: multiField
        });
        return acc;
      }
    }
    return acc;
  }, []);
};
exports.getModelIdFields = getModelIdFields;
const fetchFields = async (client, indices) => {
  const fieldCapabilities = await client.asCurrentUser.fieldCaps({
    fields: '*',
    filters: '-metadata',
    include_unmapped: true,
    index: indices
  });
  const modelIdFields = getModelIdFields(fieldCapabilities);
  const indicesAggsMappings = await Promise.all(indices.map(async index => ({
    index,
    doc: await client.asCurrentUser.search({
      index,
      size: 0,
      aggs: modelIdFields.reduce((sum, modelIdField) => ({
        ...sum,
        [modelIdField.path]: {
          terms: {
            field: modelIdField.aggField,
            size: 1
          }
        }
      }), {})
    }),
    mapping: await client.asCurrentUser.indices.getMapping({
      index
    })
  })));
  return parseFieldsCapabilities(fieldCapabilities, indicesAggsMappings);
};
exports.fetchFields = fetchFields;
const INFERENCE_MODEL_FIELD_REGEXP = /\.predicted_value|\.tokens/;
const getSemanticField = (field, semanticFields) => {
  return semanticFields.find(sf => sf.field === field);
};
const getModelField = (field, modelIdFields) => {
  var _modelIdFields$find, _modelIdFields$find2;
  // For input_output inferred fields, the model_id is at the top level
  const topLevelModelField = (_modelIdFields$find = modelIdFields.find(modelIdField => modelIdField.field === 'model_id')) === null || _modelIdFields$find === void 0 ? void 0 : _modelIdFields$find.modelId;
  if (topLevelModelField) {
    return topLevelModelField;
  }
  return (_modelIdFields$find2 = modelIdFields.find(modelIdField => modelIdField.field === field.replace(INFERENCE_MODEL_FIELD_REGEXP, '.model_id'))) === null || _modelIdFields$find2 === void 0 ? void 0 : _modelIdFields$find2.modelId;
};
const isFieldNested = (field, fieldCapsResponse) => {
  const parts = field.split('.');
  const parents = [];
  const {
    fields
  } = fieldCapsResponse;

  // Iteratively construct parent strings
  for (let i = parts.length - 1; i >= 1; i--) {
    parents.push(parts.slice(0, i).join('.'));
  }

  // Check if any of the parents are nested
  for (const parent of parents) {
    if (fields[parent] && fields[parent].nested && fields[parent].nested.type === 'nested') {
      return parent;
    }
  }
  return false;
};
const isFieldInIndex = (field, fieldKey, index) => {
  var _field$fieldKey, _field$fieldKey$indic;
  return fieldKey in field && ('indices' in field[fieldKey] ? (_field$fieldKey = field[fieldKey]) === null || _field$fieldKey === void 0 ? void 0 : (_field$fieldKey$indic = _field$fieldKey.indices) === null || _field$fieldKey$indic === void 0 ? void 0 : _field$fieldKey$indic.includes(index) : true);
};
const sortFields = fields => {
  const entries = Object.entries(fields);
  entries.sort((a, b) => a[0].localeCompare(b[0]));
  return Object.fromEntries(entries);
};
const isFieldsInMappings = mappingProperties => {
  return 'fields' in mappingProperties ? mappingProperties.fields : undefined;
};
const isPropertiesInMappings = mappingProperties => {
  return 'properties' in mappingProperties ? mappingProperties.properties : undefined;
};
const createSemanticTextFieldsList = (mappingProperties, semanticFieldsList = [], semanticTextField) => {
  Object.keys(mappingProperties || {}).forEach(field => {
    if (mappingProperties[field].type === 'semantic_text') {
      var _mapping$model_settin;
      const mapping = mappingProperties[field];
      const semanticField = {
        field: semanticTextField ? `${semanticTextField}.${field}` : field,
        inferenceId: mapping === null || mapping === void 0 ? void 0 : mapping.inference_id,
        embeddingType: mapping !== null && mapping !== void 0 && (_mapping$model_settin = mapping.model_settings) !== null && _mapping$model_settin !== void 0 && _mapping$model_settin.task_type ? EMBEDDING_TYPE[mapping.model_settings.task_type] : undefined
      };
      semanticFieldsList.push(semanticField);
      return semanticFieldsList;
    } else {
      // to handle object and multi-field type mappings
      let fieldsOrProperties = {};
      const fieldsInMappings = isFieldsInMappings(mappingProperties[field]);
      if (fieldsInMappings !== undefined) {
        fieldsOrProperties = fieldsInMappings;
      }
      const propertiesInMappings = isPropertiesInMappings(mappingProperties[field]);
      if (propertiesInMappings !== undefined) {
        fieldsOrProperties = propertiesInMappings;
      }
      createSemanticTextFieldsList(fieldsOrProperties, semanticFieldsList, field);
    }
  });
  return semanticFieldsList;
};
const parseFieldsCapabilities = (fieldCapsResponse, aggMappingDocs) => {
  const {
    indices: indexOrIndices
  } = fieldCapsResponse;
  const fields = sortFields(fieldCapsResponse.fields);
  const indices = Array.isArray(indexOrIndices) ? indexOrIndices : [indexOrIndices];
  const indexModelIdFields = aggMappingDocs.map(aggDoc => {
    const modelIdFields = Object.keys(aggDoc.doc.aggregations || {}).map(field => {
      var _field, _field$buckets, _field$buckets$;
      return {
        field,
        modelId: (_field = aggDoc.doc.aggregations[field]) === null || _field === void 0 ? void 0 : (_field$buckets = _field.buckets) === null || _field$buckets === void 0 ? void 0 : (_field$buckets$ = _field$buckets[0]) === null || _field$buckets$ === void 0 ? void 0 : _field$buckets$.key
      };
    });
    const mappingProperties = aggDoc.mapping[aggDoc.index].mappings.properties || {};
    const semanticTextFields = createSemanticTextFieldsList(mappingProperties);
    return {
      index: aggDoc.index,
      fields: modelIdFields,
      semanticTextFields
    };
  });
  const indicesFieldsMap = indices.reduce((acc, index) => {
    acc[index] = {
      elser_query_fields: [],
      dense_vector_query_fields: [],
      bm25_query_fields: [],
      source_fields: [],
      skipped_fields: 0,
      semantic_fields: []
    };
    return acc;
  }, {});

  // metadata fields that are ignored
  const shouldIgnoreField = field => {
    return !field.endsWith('model_id');
  };
  const querySourceFields = Object.keys(fields).reduce((acc, fieldKey) => {
    const field = fields[fieldKey];
    // if the field is present in all indices, the indices property is not present
    const indicesPresentIn = 'unmapped' in field ? indices.filter(index => !field.unmapped.indices.includes(index)) : indices;
    for (const index of indicesPresentIn) {
      const {
        fields: modelIdFields,
        semanticTextFields
      } = indexModelIdFields.find(indexModelIdField => indexModelIdField.index === index);
      const nestedField = isFieldNested(fieldKey, fieldCapsResponse);
      const semanticFieldMapping = getSemanticField(fieldKey, semanticTextFields);
      if (isFieldInIndex(field, 'text', index) && semanticFieldMapping) {
        // only use this when embeddingType and inferenceId is defined
        // this requires semantic_text field to be set up correctly and ingested
        if (semanticFieldMapping && semanticFieldMapping.embeddingType && semanticFieldMapping.inferenceId && !nestedField) {
          const semanticField = {
            field: fieldKey,
            inferenceId: semanticFieldMapping.inferenceId,
            embeddingType: semanticFieldMapping.embeddingType,
            indices: field.text.indices || indicesPresentIn
          };
          acc[index].semantic_fields.push(semanticField);
          acc[index].source_fields.push(fieldKey);
        } else {
          acc[index].skipped_fields++;
        }
      } else if (isFieldInIndex(field, 'sparse_vector', index)) {
        const modelId = getModelField(fieldKey, modelIdFields);
        const fieldCapabilities = field.sparse_vector;

        // Check if the sparse vector field has a model_id associated with it
        // skip this field if has no model associated with it
        // and the vectors were embedded outside of stack
        if (modelId && !nestedField) {
          const elserModelField = {
            field: fieldKey,
            model_id: modelId,
            indices: fieldCapabilities.indices || indicesPresentIn,
            // we must use sparse_vector query
            sparse_vector: true
          };
          acc[index].elser_query_fields.push(elserModelField);
        } else {
          acc[index].skipped_fields++;
        }
      } else if (isFieldInIndex(field, 'rank_features', index)) {
        const modelId = getModelField(fieldKey, modelIdFields);
        const fieldCapabilities = field.rank_features;

        // Check if the sparse vector field has a model_id associated with it
        // skip this field if has no model associated with it
        // and the vectors were embedded outside of stack
        if (modelId && !nestedField) {
          const elserModelField = {
            field: fieldKey,
            model_id: modelId,
            indices: fieldCapabilities.indices || indicesPresentIn,
            // we must use text_expansion query
            sparse_vector: false
          };
          acc[index].elser_query_fields.push(elserModelField);
        } else {
          acc[index].skipped_fields++;
        }
      } else if (isFieldInIndex(field, 'dense_vector', index)) {
        const modelId = getModelField(fieldKey, modelIdFields);
        const fieldCapabilities = field.dense_vector;

        // Check if the dense vector field has a model_id associated with it
        // skip this field if has no model associated with it
        // and the vectors were embedded outside of stack
        if (modelId && !nestedField) {
          const denseVectorField = {
            field: fieldKey,
            model_id: modelId,
            indices: fieldCapabilities.indices || indicesPresentIn
          };
          acc[index].dense_vector_query_fields.push(denseVectorField);
        } else {
          acc[index].skipped_fields++;
        }
      } else if (shouldIgnoreField(fieldKey)) {
        acc[index].source_fields.push(fieldKey);
        if ('text' in field && field.text.searchable) {
          acc[index].bm25_query_fields.push(fieldKey);
        }
      } else {
        if (fieldKey !== '_id' && fieldKey !== '_index' && fieldKey !== '_type') {
          acc[index].skipped_fields++;
        }
      }
    }
    return acc;
  }, indicesFieldsMap);
  return querySourceFields;
};
exports.parseFieldsCapabilities = parseFieldsCapabilities;