"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.cleanCachedNotebook = cleanCachedNotebook;
exports.cleanCachedNotebookCatalog = cleanCachedNotebookCatalog;
exports.cleanNotebookMetadata = cleanNotebookMetadata;
exports.createNotebooksCache = createNotebooksCache;
exports.dateWithinTTL = dateWithinTTL;
exports.isCachedNotebookCatalog = isCachedNotebookCatalog;
exports.notebookCatalogResponse = notebookCatalogResponse;
exports.validateRemoteNotebookCatalog = validateRemoteNotebookCatalog;
var _types = require("../common/types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function createNotebooksCache() {
  return {
    notebooks: {}
  };
}
function dateWithinTTL(value, ttl) {
  return (Date.now() - value.getTime()) / 1000 <= ttl;
}
function cleanCachedNotebookCatalog(notebooks) {
  const cleanedNotebooks = notebooks.map(cleanNotebookMetadata);
  return {
    notebooks: cleanedNotebooks
  };
}
function cleanCachedNotebook(notebook) {
  const {
    timestamp,
    ...result
  } = notebook;
  return result;
}
function cleanNotebookMetadata(nb) {
  const {
    id,
    title,
    description,
    link
  } = nb;
  return {
    description,
    id,
    link,
    title
  };
}
function isCachedNotebookCatalog(catalog) {
  return 'timestamp' in catalog;
}
const DEFAULT_NOTEBOOK_LIST_KEY = 'default';
function notebookCatalogResponse(catalog, list = DEFAULT_NOTEBOOK_LIST_KEY) {
  if (!catalog.lists) {
    return isCachedNotebookCatalog(catalog) ? cleanCachedNotebookCatalog(catalog.notebooks) : catalog;
  }
  const listOfNotebookIds = getListOfNotebookIds(catalog.lists, list);
  const notebookIndexMap = catalog.notebooks.reduce((indexMap, nb, i) => {
    indexMap[nb.id] = i;
    return indexMap;
  }, {});
  const notebooks = listOfNotebookIds.map(id => {
    var _catalog$notebooks$nb;
    const nbIndex = notebookIndexMap[id];
    if (nbIndex === undefined) return undefined;
    return (_catalog$notebooks$nb = catalog.notebooks[nbIndex]) !== null && _catalog$notebooks$nb !== void 0 ? _catalog$notebooks$nb : undefined;
  }).filter(nbInfo => nbInfo !== undefined);
  return cleanCachedNotebookCatalog(notebooks);
}
function getListOfNotebookIds(catalogLists, list) {
  if (list in catalogLists && catalogLists[list]) return catalogLists[list];
  if (DEFAULT_NOTEBOOK_LIST_KEY in catalogLists && catalogLists[DEFAULT_NOTEBOOK_LIST_KEY]) return catalogLists[DEFAULT_NOTEBOOK_LIST_KEY];

  // This should not happen as we should not load a catalog with lists thats missing the default list as valid,
  // but handling this case for code completeness.
  throw new Error('Notebook catalog has lists, but is missing default list'); // TODO: ?translate
}
function validateRemoteNotebookCatalog(respJson) {
  const catalog = _types.NotebookCatalogSchema.validate(respJson);
  if (catalog.lists && !(DEFAULT_NOTEBOOK_LIST_KEY in catalog.lists)) {
    // TODO: translate error message
    throw new Error('Invalid remote notebook catalog. Catalog defines lists, but is missing the default list.');
  }
  return catalog;
}