"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerJobInfoRoutesInternal = registerJobInfoRoutesInternal;
var _configSchema = require("@kbn/config-schema");
var _reportingCommon = require("@kbn/reporting-common");
var _server = require("@kbn/security-plugin/server");
var _common = require("../../common");
var _request_handler = require("../../common/request_handler");
var _jobs = require("../../common/jobs");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const {
  JOBS
} = _reportingCommon.INTERNAL_ROUTES;
function registerJobInfoRoutesInternal(reporting) {
  const setupDeps = reporting.getPluginSetupDeps();
  const {
    router
  } = setupDeps;
  const jobsQuery = (0, _jobs.jobsQueryFactory)(reporting, {
    isInternal: true
  });
  const registerInternalGetList = () => {
    // list jobs in the queue, paginated
    const path = JOBS.LIST;
    router.get({
      path,
      security: {
        authz: {
          enabled: false,
          reason: 'This route is opted out from authorization'
        }
      },
      validate: {
        query: _configSchema.schema.object({
          page: _configSchema.schema.string({
            defaultValue: '0'
          }),
          size: _configSchema.schema.string({
            defaultValue: '10'
          }),
          ids: _configSchema.schema.maybe(_configSchema.schema.string())
        })
      },
      options: {
        access: 'internal'
      }
    }, (0, _common.authorizedUserPreRouting)(reporting, async (user, context, req, res) => {
      const counters = (0, _common.getCounters)(req.route.method, path, reporting.getUsageCounter());

      // ensure the async dependencies are loaded
      if (!context.reporting) {
        return (0, _request_handler.handleUnavailable)(res);
      }
      const {
        page: queryPage = '0',
        size: querySize = '10',
        ids: queryIds = null
      } = req.query;
      const page = parseInt(queryPage, 10) || 0;
      const size = Math.min(100, parseInt(querySize, 10) || 10);
      const jobIds = queryIds ? queryIds.split(',') : null;
      const results = await jobsQuery.list(req, user, page, size, jobIds);
      counters.usageCounter();
      return res.ok({
        body: results,
        headers: {
          'content-type': 'application/json'
        }
      });
    }));
  };
  const registerInternalGetCount = () => {
    // return the count of all jobs in the queue
    const path = JOBS.COUNT;
    router.get({
      path,
      security: {
        authz: {
          enabled: false,
          reason: 'This route is opted out from authorization'
        }
      },
      validate: false,
      options: {
        access: 'internal'
      }
    }, (0, _common.authorizedUserPreRouting)(reporting, async (user, context, req, res) => {
      const counters = (0, _common.getCounters)(req.route.method, path, reporting.getUsageCounter());

      // ensure the async dependencies are loaded
      if (!context.reporting) {
        return (0, _request_handler.handleUnavailable)(res);
      }
      const count = await jobsQuery.count(user);
      counters.usageCounter();
      return res.ok({
        body: count.toString(),
        headers: {
          'content-type': 'text/plain'
        }
      });
    }));
  };

  // use common route handlers that are shared for public and internal routes
  const jobHandlers = (0, _jobs.commonJobsRouteHandlerFactory)(reporting, {
    isInternal: true
  });
  const registerInternalGetInfo = () => {
    // return some info about the job
    const path = `${JOBS.INFO_PREFIX}/{docId}`;
    router.get({
      path,
      security: {
        authz: {
          enabled: false,
          reason: 'This route is opted out from authorization'
        }
      },
      validate: jobHandlers.validate,
      options: {
        access: 'internal'
      }
    }, (0, _common.authorizedUserPreRouting)(reporting, async (user, context, req, res) => {
      const counters = (0, _common.getCounters)(req.route.method, path, reporting.getUsageCounter());

      // ensure the async dependencies are loaded
      if (!context.reporting) {
        return res.custom({
          statusCode: 503
        });
      }
      const {
        docId
      } = req.params;
      return (0, _jobs.jobManagementPreRouting)(reporting, res, docId, user, counters, {
        isInternal: true
      }, async doc => res.ok({
        body: doc,
        headers: {
          'content-type': 'application/json'
        }
      }));
    }));
  };
  const registerInternalDownloadReport = () => {
    // trigger a download of the output from a job
    const path = `${JOBS.DOWNLOAD_PREFIX}/{docId}`;
    router.get({
      path,
      security: {
        authz: {
          enabled: false,
          reason: 'This route is opted out from authorization'
        }
      },
      validate: jobHandlers.validate,
      options: {
        tags: [_server.ROUTE_TAG_CAN_REDIRECT],
        access: 'internal'
      }
    }, (0, _common.authorizedUserPreRouting)(reporting, async (user, context, req, res) => {
      return jobHandlers.handleDownloadReport({
        path,
        user,
        context,
        req,
        res
      });
    }));
  };
  const registerInternalDeleteReport = () => {
    // allow a report to be deleted
    const path = `${JOBS.DELETE_PREFIX}/{docId}`;
    router.delete({
      path,
      security: {
        authz: {
          enabled: false,
          reason: 'This route is opted out from authorization'
        }
      },
      validate: jobHandlers.validate,
      options: {
        access: 'internal'
      }
    }, (0, _common.authorizedUserPreRouting)(reporting, async (user, context, req, res) => {
      return jobHandlers.handleDeleteReport({
        path,
        user,
        context,
        req,
        res
      });
    }));
  };
  registerInternalGetList();
  registerInternalGetCount();
  registerInternalGetInfo();
  registerInternalDownloadReport();
  registerInternalDeleteReport();
}