"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ReindexServiceWrapper = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _upgradeAssistantPkgCommon = require("@kbn/upgrade-assistant-pkg-common");
var _i18n = require("@kbn/i18n");
var _std = require("@kbn/std");
var _worker = require("./worker");
var _reindex_actions = require("./reindex_actions");
var _reindex_service = require("./reindex_service");
var _error = require("./error");
var _op_utils = require("./op_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class ReindexServiceWrapper {
  constructor({
    soClient,
    credentialStore,
    clusterClient,
    logger,
    licensing,
    security,
    version
  }) {
    (0, _defineProperty2.default)(this, "reindexWorker", void 0);
    (0, _defineProperty2.default)(this, "deps", void 0);
    this.deps = {
      credentialStore,
      logger,
      licensing,
      security,
      version
    };
    this.reindexWorker = _worker.ReindexWorker.create(soClient, credentialStore, clusterClient, logger, licensing, security, version);
    this.reindexWorker.start();
  }
  getInternalApis() {
    return {
      stop: () => this.reindexWorker.stop()
    };
  }
  getScopedClient({
    dataClient,
    request,
    savedObjects
  }) {
    const callAsCurrentUser = dataClient.asCurrentUser;
    const reindexActions = (0, _reindex_actions.reindexActionsFactory)(savedObjects, callAsCurrentUser, this.deps.logger);
    const reindexService = (0, _reindex_service.reindexServiceFactory)(callAsCurrentUser, reindexActions, this.deps.logger, this.deps.licensing, this.deps.version);
    const throwIfNoPrivileges = async (indexName, newIndexName) => {
      if (!(await reindexService.hasRequiredPrivileges([indexName, newIndexName]))) {
        throw _error.error.accessForbidden(_i18n.i18n.translate('xpack.reindexService.reindexPrivilegesErrorBatch', {
          defaultMessage: 'You do not have adequate privileges to reindex "{indexName}" to "{newIndexName}".',
          values: {
            indexName,
            newIndexName
          }
        }));
      }
    };
    const reindexOrResume = async ({
      indexName,
      newIndexName,
      reindexOptions,
      settings
    }) => {
      await throwIfNoPrivileges(indexName, newIndexName);
      const existingOp = await reindexService.findReindexOperation(indexName);

      // If the reindexOp already exists and it's paused, resume it. Otherwise create a new one.
      const reindexOp = existingOp && existingOp.attributes.status === _upgradeAssistantPkgCommon.ReindexStatus.paused ? await reindexService.resumeReindexOperation(indexName, reindexOptions) : await reindexService.createReindexOperation({
        indexName,
        newIndexName,
        opts: reindexOptions,
        settings
      });

      // Add users credentials for the worker to use
      await this.deps.credentialStore.set({
        reindexOp,
        request,
        security: this.deps.security
      });
      return reindexOp.attributes;
    };
    return {
      getBatchQueueResponse: async () => {
        const inProgressOps = await reindexActions.findAllByStatus(_upgradeAssistantPkgCommon.ReindexStatus.inProgress);
        const {
          queue
        } = (0, _op_utils.sortAndOrderReindexOperations)(inProgressOps);
        return {
          queue: queue.map(savedObject => savedObject.attributes)
        };
      },
      addToBatch: async reindexJobs => {
        const results = {
          enqueued: [],
          errors: []
        };
        await (0, _std.asyncForEach)(reindexJobs, async ({
          indexName,
          newIndexName,
          settings,
          reindexOptions
        }) => {
          try {
            const result = await reindexOrResume({
              indexName,
              newIndexName,
              settings,
              reindexOptions: {
                ...reindexOptions,
                enqueue: true
              }
            });
            results.enqueued.push(result);
          } catch (e) {
            results.errors.push({
              indexName,
              message: e.message
            });
          }
        });
        return results;
      },
      hasRequiredPrivileges: reindexService.hasRequiredPrivileges.bind(reindexService),
      reindexOrResume,
      reindex: async ({
        indexName,
        newIndexName,
        reindexOptions,
        settings
      }) => {
        await throwIfNoPrivileges(indexName, newIndexName);
        const existingOp = await reindexService.findReindexOperation(indexName);
        if (existingOp && ![_upgradeAssistantPkgCommon.ReindexStatus.cancelled, _upgradeAssistantPkgCommon.ReindexStatus.completed, _upgradeAssistantPkgCommon.ReindexStatus.failed].includes(existingOp.attributes.status)) {
          throw _error.error.reindexAlreadyInProgress(_i18n.i18n.translate('xpack.reindexService.reindexAlreadyInProgressError', {
            defaultMessage: 'A reindex operation already in-progress for {indexName}',
            values: {
              indexName
            }
          }));
        }
        const reindexOp = await reindexService.createReindexOperation({
          indexName,
          newIndexName,
          opts: reindexOptions,
          settings
        });

        // Add users credentials for the worker to use
        await this.deps.credentialStore.set({
          reindexOp,
          request,
          security: this.deps.security
        });

        // Kick the worker on this node to immediately pickup the new reindex operation.
        this.getWorker().forceRefresh();
        return reindexOp.attributes;
      },
      getStatus: async indexName => {
        const hasRequiredPrivileges = await reindexService.hasRequiredPrivileges([indexName]);
        const reindexOp = await reindexService.findReindexOperation(indexName);
        // If the user doesn't have privileges than querying for warnings is going to fail.
        const warnings = hasRequiredPrivileges ? await reindexService.detectReindexWarnings(indexName) : [];
        const isTruthy = value => value === true || value === 'true';
        const {
          aliases,
          settings,
          isInDataStream,
          isFollowerIndex
        } = await reindexService.getIndexInfo(indexName);
        const body = {
          reindexOp: reindexOp ? reindexOp.attributes : undefined,
          warnings,
          hasRequiredPrivileges,
          meta: {
            indexName,
            reindexName: reindexOp === null || reindexOp === void 0 ? void 0 : reindexOp.attributes.newIndexName,
            aliases: Object.keys(aliases),
            isFrozen: isTruthy(settings === null || settings === void 0 ? void 0 : settings.frozen),
            isReadonly: isTruthy(settings === null || settings === void 0 ? void 0 : settings.verified_read_only),
            isInDataStream,
            isFollowerIndex
          }
        };
        return body;
      },
      cancel: async indexName => {
        return reindexService.cancelReindexing(indexName);
      }
    };
  }
  cleanupReindexOperations(indexNames) {
    return this.getWorker().cleanupReindexOperations(indexNames);
  }
  getWorker() {
    if (!this.reindexWorker) {
      throw new Error('Worker unavailable');
    }
    return this.reindexWorker;
  }
  stop() {
    this.reindexWorker.stop();
  }
}
exports.ReindexServiceWrapper = ReindexServiceWrapper;