"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getIntegrationNamespaces = void 0;
var _common = require("@kbn/fleet-plugin/common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Retrieves namespaces used by specified integrations.
 * Based on the implementation from security_solution's fetchIntegrationNamespaces.
 */
const getIntegrationNamespaces = async ({
  logger,
  soClient,
  packagePolicyService,
  agentPolicyService,
  integrationNames = []
}) => {
  const integrationToNamespaceMap = integrationNames.reduce((acc, name) => {
    acc[name] = new Set();
    return acc;
  }, {});
  const agentPolicyIdsToRetrieve = {};
  if (integrationNames.length > 0) {
    const kuery = `${_common.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.name: (${integrationNames.join(' OR ')})`;
    logger.debug(`Fetch of policies for integrations using Kuery [${kuery}]`);
    const policiesFound = await packagePolicyService.list(soClient, {
      perPage: 10000,
      kuery
    });
    logger.trace(`Fetch of policies for integrations using Kuery [${kuery}] returned ${policiesFound.items.length} policies`);
    for (const packagePolicy of policiesFound.items) {
      var _packagePolicy$packag;
      if ((_packagePolicy$packag = packagePolicy.package) !== null && _packagePolicy$packag !== void 0 && _packagePolicy$packag.name) {
        const integrationName = packagePolicy.package.name;
        if (packagePolicy.namespace) {
          integrationToNamespaceMap[integrationName].add(packagePolicy.namespace);
        } else {
          // Integration policy does not have an explicit namespace, which means it
          // inherits it from the associated agent policies. We'll retrieve these next
          packagePolicy.policy_ids.forEach(agentPolicyId => {
            if (!agentPolicyIdsToRetrieve[agentPolicyId]) {
              agentPolicyIdsToRetrieve[agentPolicyId] = new Set();
            }
            agentPolicyIdsToRetrieve[agentPolicyId].add(integrationToNamespaceMap[integrationName]);
          });
        }
      }
    }
  }
  const agentPolicyIds = Object.keys(agentPolicyIdsToRetrieve);
  if (agentPolicyIds.length > 0) {
    logger.debug(`Retrieving agent policies from fleet for: ${agentPolicyIds.join(', ')}`);
    const agentPolicies = await agentPolicyService.getByIds(soClient, agentPolicyIds);
    logger.trace(`Fleet agent policies retrieved: ${agentPolicies.length} policies`);
    for (const agentPolicy of agentPolicies) {
      for (const nameSpaceSet of agentPolicyIdsToRetrieve[agentPolicy.id]) {
        nameSpaceSet.add(agentPolicy.namespace);
      }
    }
  }
  const response = Object.entries(integrationToNamespaceMap).reduce((acc, [integrationName, namespaceSet]) => {
    acc[integrationName] = Array.from(namespaceSet.values());
    return acc;
  }, {});
  logger.debug(`Integration namespaces in use: ${JSON.stringify(response)}`);
  return response;
};
exports.getIntegrationNamespaces = getIntegrationNamespaces;