"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createPersistedToolClient = exports.createPersistedProviderFn = void 0;
var _onechatCommon = require("@kbn/onechat-common");
var _definitions = require("../tool_types/definitions");
var _client = require("./client");
var _converter = require("./converter");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createPersistedProviderFn = opts => ({
  request,
  space
}) => {
  return createPersistedToolClient({
    ...opts,
    request,
    space
  });
};
exports.createPersistedProviderFn = createPersistedProviderFn;
const createPersistedToolClient = ({
  request,
  toolTypes,
  logger,
  esClient,
  space
}) => {
  const toolClient = (0, _client.createClient)({
    space,
    esClient,
    logger
  });
  const definitionMap = toolTypes.filter(_definitions.isEnabledDefinition).reduce((map, def) => {
    map[def.toolType] = def;
    return map;
  }, {});
  const validationContext = () => {
    return {
      esClient,
      request,
      spaceId: space
    };
  };
  const conversionContext = () => {
    return {
      esClient,
      request,
      spaceId: space
    };
  };
  return {
    id: 'persisted',
    readonly: false,
    async has(toolId) {
      try {
        await toolClient.get(toolId);
        return true;
      } catch (e) {
        if ((0, _onechatCommon.isToolNotFoundError)(e)) {
          return false;
        }
        throw e;
      }
    },
    async get(toolId) {
      const tool = await toolClient.get(toolId);
      const definition = definitionMap[tool.type];
      if (!definition) {
        throw (0, _onechatCommon.createBadRequestError)(`Unknown type for tool '${toolId}': '${tool.type}'`);
      }
      return (0, _converter.convertPersistedDefinition)({
        tool,
        definition,
        context: conversionContext()
      });
    },
    async list() {
      const tools = await toolClient.list();
      const context = conversionContext();
      return tools.filter(tool => {
        // evict unknown tools - atm it's used for workflow tools if the plugin is disabled.
        return definitionMap[tool.type];
      }).map(tool => {
        const definition = definitionMap[tool.type];
        return (0, _converter.convertPersistedDefinition)({
          tool,
          definition,
          context
        });
      });
    },
    async create(createRequest) {
      const definition = definitionMap[createRequest.type];
      if (!definition) {
        throw (0, _onechatCommon.createBadRequestError)(`Unknown tool type: '${createRequest.type}'`);
      }
      try {
        definition.createSchema.validate(createRequest.configuration);
      } catch (e) {
        throw (0, _onechatCommon.createBadRequestError)(`Invalid configuration for tool type ${createRequest.type}: ${e.message}`);
      }
      let updatedConfig;
      try {
        updatedConfig = await definition.validateForCreate({
          config: createRequest.configuration,
          context: validationContext()
        });
      } catch (e) {
        throw (0, _onechatCommon.createBadRequestError)(`Invalid configuration for tool type ${createRequest.type}: ${e.message}`);
      }
      const mergedRequest = {
        ...createRequest,
        configuration: updatedConfig
      };
      const tool = await toolClient.create(mergedRequest);
      return (0, _converter.convertPersistedDefinition)({
        tool,
        definition,
        context: conversionContext()
      });
    },
    async update(toolId, updateRequest) {
      const existingTool = await this.get(toolId);
      const definition = definitionMap[existingTool.type];
      if (!definition) {
        throw (0, _onechatCommon.createBadRequestError)(`Unknown tool type: '${existingTool.type}'`);
      }
      try {
        definition.updateSchema.validate(updateRequest.configuration);
      } catch (e) {
        throw (0, _onechatCommon.createBadRequestError)(`Invalid configuration for tool type ${existingTool.type}: ${e.message}`);
      }
      let updatedConfig;
      try {
        var _updateRequest$config;
        updatedConfig = await definition.validateForUpdate({
          update: (_updateRequest$config = updateRequest.configuration) !== null && _updateRequest$config !== void 0 ? _updateRequest$config : {},
          current: existingTool.configuration,
          context: validationContext()
        });
      } catch (e) {
        throw (0, _onechatCommon.createBadRequestError)(`Invalid configuration for tool type ${existingTool.type}: ${e.message}`);
      }
      const mergedConfig = {
        ...updateRequest,
        configuration: updatedConfig
      };
      const tool = await toolClient.update(toolId, mergedConfig);
      return (0, _converter.convertPersistedDefinition)({
        tool,
        definition,
        context: conversionContext()
      });
    },
    async delete(toolId) {
      return toolClient.delete(toolId);
    }
  };
};
exports.createPersistedToolClient = createPersistedToolClient;