"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDocumentByIdTool = void 0;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _onechatGenaiUtils = require("@kbn/onechat-genai-utils");
var _onechatServer = require("@kbn/onechat-server");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getDocumentByIdSchema = _zod.z.object({
  id: _zod.z.string().describe('ID of the document to retrieve'),
  index: _zod.z.string().describe('Name of the index to retrieve the document from')
});
const getDocumentByIdTool = () => {
  return {
    id: _onechatCommon.platformCoreTools.getDocumentById,
    type: _onechatCommon.ToolType.builtin,
    description: 'Retrieve the full content (source) of an Elasticsearch document based on its ID and index name.',
    schema: getDocumentByIdSchema,
    handler: async ({
      id,
      index
    }, {
      esClient
    }) => {
      const result = await (0, _onechatGenaiUtils.getDocumentById)({
        id,
        index,
        esClient: esClient.asCurrentUser
      });
      if (result.found) {
        return {
          results: [{
            type: _tool_result.ToolResultType.resource,
            data: {
              reference: {
                id: result.id,
                index: result.index
              },
              partial: false,
              content: result._source
            }
          }]
        };
      }
      return {
        results: [(0, _onechatServer.createErrorResult)({
          message: `Document with ID '${result.id}' not found in index '${result.index}'`,
          metadata: {
            id: result.id,
            index: result.index
          }
        })]
      };
    },
    tags: []
  };
};
exports.getDocumentByIdTool = getDocumentByIdTool;