"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _uuid = require("uuid");
var _onechatCommon = require("@kbn/onechat-common");
var _spaces = require("../../utils/spaces");
var _converters = require("./converters");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createClient = ({
  space,
  storage,
  user
}) => {
  return new ConversationClientImpl({
    storage,
    user,
    space
  });
};
exports.createClient = createClient;
class ConversationClientImpl {
  constructor({
    storage,
    user,
    space
  }) {
    (0, _defineProperty2.default)(this, "space", void 0);
    (0, _defineProperty2.default)(this, "storage", void 0);
    (0, _defineProperty2.default)(this, "user", void 0);
    this.storage = storage;
    this.user = user;
    this.space = space;
  }
  async list(options = {}) {
    const {
      agentId
    } = options;
    const response = await this.storage.getClient().search({
      track_total_hits: false,
      size: 1000,
      _source: {
        excludes: ['rounds']
      },
      query: {
        bool: {
          filter: [(0, _spaces.createSpaceDslFilter)(this.space)],
          must: [{
            term: this.user.username ? {
              user_name: this.user.username
            } : {
              user_id: this.user.id
            }
          }, ...(agentId ? [{
            term: {
              agent_id: agentId
            }
          }] : [])]
        }
      }
    });
    return response.hits.hits.map(hit => (0, _converters.fromEsWithoutRounds)(hit));
  }
  async get(conversationId) {
    const document = await this._get(conversationId);
    if (!document) {
      throw (0, _onechatCommon.createConversationNotFoundError)({
        conversationId
      });
    }
    if (!hasAccess({
      conversation: document,
      user: this.user
    })) {
      throw (0, _onechatCommon.createConversationNotFoundError)({
        conversationId
      });
    }
    return (0, _converters.fromEs)(document);
  }
  async exists(conversationId) {
    const document = await this._get(conversationId);
    if (!document) {
      return false;
    }
    return hasAccess({
      conversation: document,
      user: this.user
    });
  }
  async create(conversation) {
    var _conversation$id;
    const now = new Date();
    const id = (_conversation$id = conversation.id) !== null && _conversation$id !== void 0 ? _conversation$id : (0, _uuid.v4)();
    const attributes = (0, _converters.createRequestToEs)({
      conversation,
      currentUser: this.user,
      creationDate: now,
      space: this.space
    });
    await this.storage.getClient().index({
      id,
      document: attributes
    });
    return this.get(id);
  }
  async update(conversationUpdate) {
    const {
      id: conversationId
    } = conversationUpdate;
    const now = new Date();
    const document = await this._get(conversationUpdate.id);
    if (!document) {
      throw (0, _onechatCommon.createConversationNotFoundError)({
        conversationId
      });
    }
    if (!hasAccess({
      conversation: document,
      user: this.user
    })) {
      throw (0, _onechatCommon.createConversationNotFoundError)({
        conversationId
      });
    }
    const storedConversation = (0, _converters.fromEs)(document);
    const updatedConversation = (0, _converters.updateConversation)({
      conversation: storedConversation,
      update: conversationUpdate,
      updateDate: now,
      space: this.space
    });
    const attributes = (0, _converters.toEs)(updatedConversation, this.space);
    await this.storage.getClient().index({
      id: conversationUpdate.id,
      document: attributes
    });
    return this.get(conversationUpdate.id);
  }
  async delete(conversationId) {
    const document = await this._get(conversationId);
    if (!document) {
      throw (0, _onechatCommon.createConversationNotFoundError)({
        conversationId
      });
    }
    if (!hasAccess({
      conversation: document,
      user: this.user
    })) {
      throw (0, _onechatCommon.createConversationNotFoundError)({
        conversationId
      });
    }
    const {
      result
    } = await this.storage.getClient().delete({
      id: conversationId
    });
    return result === 'deleted';
  }
  async _get(conversationId) {
    const response = await this.storage.getClient().search({
      track_total_hits: false,
      size: 1,
      terminate_after: 1,
      query: {
        bool: {
          filter: [(0, _spaces.createSpaceDslFilter)(this.space), {
            term: {
              _id: conversationId
            }
          }]
        }
      }
    });
    if (response.hits.hits.length === 0) {
      return undefined;
    } else {
      return response.hits.hits[0];
    }
  }
}
const hasAccess = ({
  conversation,
  user
}) => {
  return conversation._source.user_id === user.id || conversation._source.user_name === user.username;
};