"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ChunkType = exports.ChunkDecisionBuffer = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
/**
 * Defines the types of content chunks the buffer can classify.
 */
let ChunkType = exports.ChunkType = /*#__PURE__*/function (ChunkType) {
  ChunkType["FinalAnswer"] = "final_answer";
  ChunkType["ToolReasoning"] = "tool_reasoning";
  return ChunkType;
}({});
/**
 * The structured output for a processed chunk.
 */
/**
 * A utility to buffer and conditionally classify text chunks from an LLM stream.
 */
class ChunkDecisionBuffer {
  constructor({
    tag
  }) {
    (0, _defineProperty2.default)(this, "openingTag", void 0);
    (0, _defineProperty2.default)(this, "closingTag", void 0);
    (0, _defineProperty2.default)(this, "decisionThreshold", void 0);
    (0, _defineProperty2.default)(this, "decisionMade", false);
    (0, _defineProperty2.default)(this, "isToolTurn", false);
    (0, _defineProperty2.default)(this, "buffer", []);
    (0, _defineProperty2.default)(this, "accumulatedText", '');
    (0, _defineProperty2.default)(this, "trailingCleanBuffer", '');
    if (!tag.startsWith('<') || !tag.endsWith('>')) {
      throw new Error('Tag must be a valid XML-like tag, e.g., "<example>"');
    }
    this.openingTag = tag;
    this.closingTag = `</${tag.substring(1)}`;
    this.decisionThreshold = this.openingTag.length;
  }
  reset() {
    this.decisionMade = false;
    this.isToolTurn = false;
    this.buffer = [];
    this.accumulatedText = '';
    this.trailingCleanBuffer = '';
  }
  process(chunkText) {
    if (this.decisionMade) {
      const type = this.isToolTurn ? ChunkType.ToolReasoning : ChunkType.FinalAnswer;
      const cleanedText = this.cleanAndManageBuffer(chunkText);
      return cleanedText ? [{
        type,
        text: cleanedText
      }] : null;
    }
    this.buffer.push(chunkText);
    this.accumulatedText += chunkText;
    if (this.accumulatedText.trimStart().startsWith(this.openingTag)) {
      this.decisionMade = true;
      this.isToolTurn = true;
      return this.flushAndClassifyBuffer(ChunkType.ToolReasoning);
    }
    if (this.accumulatedText.length >= this.decisionThreshold) {
      this.decisionMade = true;
      this.isToolTurn = false;
      return this.flushAndClassifyBuffer(ChunkType.FinalAnswer);
    }
    return null;
  }
  flush() {
    if (!this.decisionMade && this.buffer.length > 0) {
      const fullText = this.buffer.join('');
      this.buffer = [];
      if (fullText) {
        return [{
          type: ChunkType.FinalAnswer,
          text: fullText
        }];
      }
      return null;
    }
    let remainingText = this.trailingCleanBuffer;
    this.trailingCleanBuffer = '';
    if (remainingText) {
      // At the very end of the stream, we must clean up any partial tags.
      const lastTagStart = remainingText.lastIndexOf('<');
      if (lastTagStart !== -1) {
        const potentialFragment = remainingText.substring(lastTagStart);
        if (this.openingTag.startsWith(potentialFragment) || this.closingTag.startsWith(potentialFragment)) {
          // It is a partial tag, so we trim it off before flushing.
          remainingText = remainingText.substring(0, lastTagStart);
        }
      }
      if (remainingText) {
        const type = this.isToolTurn ? ChunkType.ToolReasoning : ChunkType.FinalAnswer;
        return [{
          type,
          text: remainingText
        }];
      }
    }
    return null;
  }
  flushAndClassifyBuffer(type) {
    const combinedText = this.buffer.join('');
    this.buffer = [];
    const cleanedText = this.cleanAndManageBuffer(combinedText);
    return cleanedText ? [{
      type,
      text: cleanedText
    }] : [];
  }

  /**
   * A stream-safe method to clean tags from text using pattern matching.
   * This is more robust than the previous length-based heuristic.
   */
  cleanAndManageBuffer(newText) {
    let textToProcess = this.trailingCleanBuffer + newText;

    // Replace any complete tags within the current working text
    textToProcess = textToProcess.replace(this.openingTag, '').replace(this.closingTag, '');

    // Now, decide what part of textToProcess is safe to emit.
    let safeToEmit = textToProcess;

    // Find the last potential start of a tag ('<')
    const lastTagStart = textToProcess.lastIndexOf('<');
    if (lastTagStart !== -1) {
      const potentialFragment = textToProcess.substring(lastTagStart);
      // Check if this fragment could be the start of either of our tags
      if (this.openingTag.startsWith(potentialFragment) || this.closingTag.startsWith(potentialFragment)) {
        // It's a potential partial tag. Hold it back for the next chunk.
        safeToEmit = textToProcess.substring(0, lastTagStart);
        this.trailingCleanBuffer = potentialFragment;
      } else {
        // It's a '<' but not part of our tags, so clear the trailing buffer
        this.trailingCleanBuffer = '';
      }
    } else {
      this.trailingCleanBuffer = '';
    }
    return safeToEmit;
  }
}
exports.ChunkDecisionBuffer = ChunkDecisionBuffer;