"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useConversationActions = void 0;
var _onechatCommon = require("@kbn/onechat-common");
var _reactQuery = require("@tanstack/react-query");
var _immer = _interopRequireDefault(require("immer"));
var _react = require("react");
var _useLocalStorage = _interopRequireDefault(require("react-use/lib/useLocalStorage"));
var _query_keys = require("../query_keys");
var _storage_keys = require("../storage_keys");
var _app_paths = require("../utils/app_paths");
var _new_conversation = require("../utils/new_conversation");
var _use_conversation_id = require("./use_conversation_id");
var _use_navigation = require("./use_navigation");
var _use_onechat_service = require("./use_onechat_service");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const pendingRoundId = '__pending__';
const useConversationActions = () => {
  const queryClient = (0, _reactQuery.useQueryClient)();
  const conversationId = (0, _use_conversation_id.useConversationId)();
  const [, setAgentIdStorage] = (0, _useLocalStorage.default)(_storage_keys.storageKeys.agentId);
  const {
    conversationsService
  } = (0, _use_onechat_service.useOnechatServices)();
  const queryKey = _query_keys.queryKeys.conversations.byId(conversationId !== null && conversationId !== void 0 ? conversationId : _new_conversation.newConversationId);
  const setConversation = updater => {
    queryClient.setQueryData(queryKey, updater);
  };
  const setCurrentRound = updater => {
    setConversation((0, _immer.default)(draft => {
      var _draft$rounds;
      const round = draft === null || draft === void 0 ? void 0 : (_draft$rounds = draft.rounds) === null || _draft$rounds === void 0 ? void 0 : _draft$rounds.at(-1);
      if (round) {
        updater(round);
      }
    }));
  };
  const {
    navigateToOnechatUrl
  } = (0, _use_navigation.useNavigation)();
  const shouldAllowConversationRedirectRef = (0, _react.useRef)(true);
  (0, _react.useEffect)(() => {
    return () => {
      // On unmount disable conversation redirect
      shouldAllowConversationRedirectRef.current = false;
    };
  }, []);
  const navigateToConversation = ({
    nextConversationId
  }) => {
    // Navigate to the conversation if redirect is allowed
    if (shouldAllowConversationRedirectRef.current) {
      const path = _app_paths.appPaths.chat.conversation({
        conversationId: nextConversationId
      });
      const params = undefined;
      const state = {
        shouldStickToBottom: false
      };
      navigateToOnechatUrl(path, params, state);
    }
  };
  return {
    removeNewConversationQuery: () => {
      queryClient.removeQueries({
        queryKey: _query_keys.queryKeys.conversations.byId(_new_conversation.newConversationId)
      });
    },
    invalidateConversation: () => {
      queryClient.invalidateQueries({
        queryKey
      });
    },
    addOptimisticRound: ({
      userMessage
    }) => {
      setConversation((0, _immer.default)(draft => {
        const nextRound = {
          id: pendingRoundId,
          input: {
            message: userMessage
          },
          response: {
            message: ''
          },
          steps: []
        };
        if (!draft) {
          const newConversation = (0, _new_conversation.createNewConversation)();
          newConversation.rounds.push(nextRound);
          return newConversation;
        }
        draft.rounds.push(nextRound);
      }));
    },
    removeOptimisticRound: () => {
      setConversation((0, _immer.default)(draft => {
        var _draft$rounds2;
        draft === null || draft === void 0 ? void 0 : (_draft$rounds2 = draft.rounds) === null || _draft$rounds2 === void 0 ? void 0 : _draft$rounds2.pop();
      }));
    },
    setAgentId: agentId => {
      // We allow to change agent only at the start of the conversation
      if (conversationId) {
        return;
      }
      setConversation((0, _immer.default)(draft => {
        if (!draft) {
          const newConversation = (0, _new_conversation.createNewConversation)();
          newConversation.agent_id = agentId;
          return newConversation;
        }
        draft.agent_id = agentId;
      }));
      setAgentIdStorage(agentId);
    },
    addReasoningStep: ({
      step
    }) => {
      setCurrentRound(round => {
        round.steps.push(step);
      });
    },
    addToolCall: ({
      step
    }) => {
      setCurrentRound(round => {
        round.steps.push(step);
      });
    },
    setToolCallProgress: ({
      progress,
      toolCallId
    }) => {
      setCurrentRound(round => {
        const step = round.steps.filter(_onechatCommon.isToolCallStep).find(s => s.tool_call_id === toolCallId);
        if (step) {
          if (!step.progression) {
            step.progression = [];
          }
          step.progression.push(progress);
        }
      });
    },
    setToolCallResult: ({
      results,
      toolCallId
    }) => {
      setCurrentRound(round => {
        const step = round.steps.filter(_onechatCommon.isToolCallStep).find(s => s.tool_call_id === toolCallId);
        if (step) {
          step.results = results;
        }
      });
    },
    setAssistantMessage: ({
      assistantMessage
    }) => {
      setCurrentRound(round => {
        round.response.message = assistantMessage;
      });
    },
    addAssistantMessageChunk: ({
      messageChunk
    }) => {
      setCurrentRound(round => {
        round.response.message += messageChunk;
      });
    },
    onConversationCreated: ({
      conversationId: id,
      title
    }) => {
      const current = queryClient.getQueryData(queryKey);
      if (!current) {
        throw new Error('Conversation not created');
      }
      // 1. Update individual conversation cache (with rounds)
      queryClient.setQueryData(_query_keys.queryKeys.conversations.byId(id), (0, _immer.default)(current, draft => {
        draft.id = id;
        draft.title = title;
      }));
      // 2. Invalidate conversation list to get updated data from server - this updates the conversations view in the sidebar
      queryClient.invalidateQueries({
        queryKey: _query_keys.queryKeys.conversations.all
      });
      navigateToConversation({
        nextConversationId: id
      });
    },
    deleteConversation: async id => {
      await conversationsService.delete({
        conversationId: id
      });

      // Check if we're deleting the current conversation
      const isCurrentConversation = conversationId === id;
      if (isCurrentConversation) {
        // If deleting current conversation, navigate to new conversation
        const path = _app_paths.appPaths.chat.new;
        navigateToOnechatUrl(path, undefined, {
          shouldStickToBottom: true
        });
      }
      // If deleting other conversations, stay at current conversation (no navigation needed)

      queryClient.removeQueries({
        queryKey: _query_keys.queryKeys.conversations.byId(id)
      });
      queryClient.invalidateQueries({
        queryKey: _query_keys.queryKeys.conversations.all
      });
    }
  };
};
exports.useConversationActions = useConversationActions;