"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.KB_REINDEXING_LOCK_ID = void 0;
exports.getActiveReindexingTaskId = getActiveReindexingTaskId;
exports.getNextWriteIndexName = getNextWriteIndexName;
exports.isReIndexInProgress = isReIndexInProgress;
exports.reIndexKnowledgeBaseWithLock = reIndexKnowledgeBaseWithLock;
var _lodash = require("lodash");
var _pRetry = _interopRequireDefault(require("p-retry"));
var _lockManager = require("@kbn/lock-manager");
var _ = require("..");
var _create_knowledge_base_index = require("./create_knowledge_base_index");
var _update_knowledge_base_index_alias = require("./update_knowledge_base_index_alias");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const KB_REINDEXING_LOCK_ID = exports.KB_REINDEXING_LOCK_ID = 'observability_ai_assistant:kb_reindexing';
async function reIndexKnowledgeBaseWithLock({
  core,
  logger,
  esClient
}) {
  const lmService = new _lockManager.LockManagerService(core, logger);
  return lmService.withLock(KB_REINDEXING_LOCK_ID, () => reIndexKnowledgeBase({
    logger,
    esClient
  }));
}
async function reIndexKnowledgeBase({
  logger,
  esClient
}) {
  var _reindexResponse$task;
  const activeReindexingTask = await getActiveReindexingTaskId(esClient);
  if (activeReindexingTask) {
    throw new Error(`Re-indexing task "${activeReindexingTask}" is already in progress for the knowledge base index: ${_.resourceNames.writeIndexAlias.kb}`);
  }
  const {
    currentWriteIndexName,
    nextWriteIndexName
  } = await getCurrentAndNextWriteIndexNames({
    esClient,
    logger
  });
  await (0, _create_knowledge_base_index.createKnowledgeBaseIndex)({
    esClient,
    logger,
    indexName: nextWriteIndexName
  });
  logger.info(`Re-indexing knowledge base from "${currentWriteIndexName}" to index "${nextWriteIndexName}"...`);

  // Point write index alias to the new index
  await (0, _update_knowledge_base_index_alias.updateKnowledgeBaseWriteIndexAlias)({
    esClient,
    logger,
    nextWriteIndexName,
    currentWriteIndexName
  });
  const reindexResponse = await esClient.asInternalUser.reindex({
    source: {
      index: currentWriteIndexName
    },
    dest: {
      index: nextWriteIndexName
    },
    refresh: true,
    wait_for_completion: false
  });
  const taskId = (_reindexResponse$task = reindexResponse.task) === null || _reindexResponse$task === void 0 ? void 0 : _reindexResponse$task.toString();
  if (taskId) {
    await waitForReIndexTaskToComplete({
      esClient,
      taskId,
      logger
    });
  } else {
    throw new Error(`ID for re-indexing task was not found`);
  }

  // Delete original index
  logger.debug(`Deleting write index "${currentWriteIndexName}"`);
  await esClient.asInternalUser.indices.delete({
    index: currentWriteIndexName
  });
}
async function getCurrentWriteIndexName(esClient) {
  var _Object$entries$find;
  const response = await esClient.asInternalUser.indices.getAlias({
    name: _.resourceNames.writeIndexAlias.kb
  }, {
    ignore: [404]
  });
  const currentWriteIndexName = (_Object$entries$find = Object.entries(response).find(([, aliasInfo]) => {
    var _aliasInfo$aliases$re;
    return (_aliasInfo$aliases$re = aliasInfo.aliases[_.resourceNames.writeIndexAlias.kb]) === null || _aliasInfo$aliases$re === void 0 ? void 0 : _aliasInfo$aliases$re.is_write_index;
  })) === null || _Object$entries$find === void 0 ? void 0 : _Object$entries$find[0];
  return currentWriteIndexName;
}
function getNextWriteIndexName(currentWriteIndexName) {
  if (!currentWriteIndexName) {
    return;
  }
  const latestIndexNumber = (0, _lodash.last)(currentWriteIndexName.split('-'));
  if (!latestIndexNumber) {
    return;
  }

  // sequence number must be a six digit zero padded number like 000008 or 002201
  const isSequenceNumberValid = /^\d{6}$/.test(latestIndexNumber);
  if (!isSequenceNumberValid) {
    return;
  }
  const nextIndexSequenceNumber = (parseInt(latestIndexNumber, 10) + 1).toString().padStart(6, '0');
  return `${_.resourceNames.writeIndexAlias.kb}-${nextIndexSequenceNumber}`;
}
async function getCurrentAndNextWriteIndexNames({
  esClient,
  logger
}) {
  const currentWriteIndexName = await getCurrentWriteIndexName(esClient);
  const nextWriteIndexName = getNextWriteIndexName(currentWriteIndexName);
  if (!currentWriteIndexName || !nextWriteIndexName) {
    throw new Error(`"${currentWriteIndexName}" is not a valid write index name. Skipping re-indexing of knowledge base.`);
  }
  return {
    currentWriteIndexName,
    nextWriteIndexName
  };
}
async function getActiveReindexingTaskId(esClient) {
  const response = await esClient.asInternalUser.tasks.list({
    detailed: true,
    actions: ['indices:data/write/reindex']
  });
  for (const node of Object.values((_response$nodes = response.nodes) !== null && _response$nodes !== void 0 ? _response$nodes : {})) {
    var _response$nodes;
    for (const [taskId, task] of Object.entries(node.tasks)) {
      var _task$description;
      if ((_task$description = task.description) !== null && _task$description !== void 0 && _task$description.includes(_.resourceNames.writeIndexAlias.kb)) {
        return taskId;
      }
    }
  }
}
async function waitForReIndexTaskToComplete({
  esClient,
  taskId,
  logger
}) {
  return (0, _pRetry.default)(async () => {
    const taskResponse = await esClient.asInternalUser.tasks.get({
      task_id: taskId,
      wait_for_completion: false
    });
    if (!taskResponse.completed) {
      logger.debug(`Waiting for re-indexing task "${taskId}" to complete...`);
      throw new Error(`Waiting for re-indexing task "${taskId}" to complete...`);
    }
  }, {
    forever: true,
    maxTimeout: 10000
  });
}
async function isReIndexInProgress({
  esClient,
  logger,
  core
}) {
  const lmService = new _lockManager.LockManagerService(core, logger);
  const [lock, activeReindexingTask] = await Promise.all([lmService.getLock(KB_REINDEXING_LOCK_ID), getActiveReindexingTaskId(esClient)]);
  return lock !== undefined || activeReindexingTask !== undefined;
}