"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.recallFromSearchConnectors = recallFromSearchConnectors;
var _lodash = require("lodash");
var _common = require("../../../common");
var _get_elser_model_id = require("./get_elser_model_id");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function recallFromSearchConnectors({
  queries,
  esClient,
  uiSettingsClient,
  logger,
  core
}) {
  const connectorIndices = await getConnectorIndices(esClient, uiSettingsClient, logger);
  logger.debug(`Found connector indices: ${connectorIndices}`);
  const [semanticTextConnectors, legacyConnectors] = await Promise.all([recallFromSemanticTextConnectors({
    queries,
    esClient,
    uiSettingsClient,
    logger,
    core,
    connectorIndices
  }), recallFromLegacyConnectors({
    queries,
    esClient,
    uiSettingsClient,
    logger,
    core,
    connectorIndices
  })]);
  return (0, _lodash.orderBy)([...semanticTextConnectors, ...legacyConnectors], entry => entry.esScore, 'desc');
}
async function recallFromSemanticTextConnectors({
  queries,
  esClient,
  logger,
  core,
  connectorIndices
}) {
  const fieldCaps = await esClient.asCurrentUser.fieldCaps({
    index: connectorIndices,
    fields: `*`,
    allow_no_indices: true,
    types: ['text'],
    filters: '-metadata,-parent'
  });
  const textFields = Object.keys(fieldCaps.fields);
  if (!textFields.length) {
    logger.debug(`No text fields found in indices: ${connectorIndices}`);
    return [];
  }
  logger.debug(`Text field for search connectors: ${textFields}`);
  const params = {
    index: connectorIndices,
    size: 20,
    query: {
      bool: {
        should: textFields.flatMap(field => {
          return queries.map(({
            text,
            boost = 1
          }) => ({
            match: {
              [field]: {
                query: text,
                boost
              }
            }
          }));
        }),
        minimum_should_match: 1
      }
    }
  };
  const response = await esClient.asCurrentUser.search(params);
  const results = response.hits.hits.map(hit => ({
    text: JSON.stringify(hit._source),
    esScore: hit._score,
    id: hit._id
  }));
  return results;
}
async function recallFromLegacyConnectors({
  queries,
  esClient,
  logger,
  core,
  connectorIndices
}) {
  const ML_INFERENCE_PREFIX = 'ml.inference.';
  const modelIdPromise = (0, _get_elser_model_id.getElserModelId)({
    core,
    logger
  }); // pre-fetch modelId in parallel with fieldCaps
  const fieldCaps = await esClient.asCurrentUser.fieldCaps({
    index: connectorIndices,
    fields: `${ML_INFERENCE_PREFIX}*`,
    allow_no_indices: true,
    types: ['sparse_vector'],
    filters: '-metadata,-parent'
  });
  const fieldsWithVectors = Object.keys(fieldCaps.fields).map(field => field.replace('_expanded.predicted_value', '').replace(ML_INFERENCE_PREFIX, ''));
  if (!fieldsWithVectors.length) {
    return [];
  }
  const modelId = await modelIdPromise;
  const esQueries = fieldsWithVectors.flatMap(field => {
    const vectorField = `${ML_INFERENCE_PREFIX}${field}_expanded.predicted_value`;
    const modelField = `${ML_INFERENCE_PREFIX}${field}_expanded.model_id`;
    return queries.map(({
      text,
      boost = 1
    }) => {
      return {
        bool: {
          should: [{
            sparse_vector: {
              field: vectorField,
              query: text,
              inference_id: modelId,
              boost
            }
          }],
          filter: [{
            term: {
              [modelField]: modelId
            }
          }]
        }
      };
    });
  });
  const response = await esClient.asCurrentUser.search({
    index: connectorIndices,
    size: 20,
    _source: {
      exclude: ['_*', 'ml*']
    },
    query: {
      bool: {
        should: esQueries
      }
    }
  });
  const results = response.hits.hits.map(hit => ({
    text: JSON.stringify(hit._source),
    esScore: hit._score,
    id: hit._id
  }));
  return results;
}
async function getConnectorIndices(esClient, uiSettingsClient, logger) {
  var _response$results;
  // improve performance by running this in parallel with the `uiSettingsClient` request
  const responsePromise = esClient.asInternalUser.connector.list({
    filter_path: 'results.index_name'
  }).catch(e => {
    logger.warn(`Failed to fetch connector indices due to ${e.message}`);
    return {
      results: []
    };
  });
  const customSearchConnectorIndex = await uiSettingsClient.get(_common.aiAssistantSearchConnectorIndexPattern);
  if (customSearchConnectorIndex) {
    return customSearchConnectorIndex.split(',');
  }
  const response = await responsePromise;
  const connectorIndices = (0, _lodash.compact)((_response$results = response.results) === null || _response$results === void 0 ? void 0 : _response$results.map(result => result.index_name));

  // preserve backwards compatibility with 8.14 (may not be needed in the future)
  if ((0, _lodash.isEmpty)(connectorIndices)) {
    return ['search-*'];
  }
  return connectorIndices;
}