"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.flattenObject = flattenObject;
exports.flattenObjectNestedLast = flattenObjectNestedLast;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Returns a flattened version of the input object also accounting for nested properties.
 * @param obj - The input object.
 * @param parentKey - The initial key used for recursive flattening.
 * @returns An object containing all the flattened properties.
 */
function flattenObject(obj, parentKey = '') {
  const result = {};
  for (const key in obj) {
    if (Object.hasOwn(obj, key)) {
      const value = obj[key];
      const newKey = parentKey ? `${parentKey}.${key}` : key;
      if ((0, _lodash.isPlainObject)(value)) {
        Object.assign(result, flattenObject(value, newKey));
      } else {
        result[newKey] = value;
      }
    }
  }
  return result;
}

/**
 * Returns a flattened version of the input object, giving higher priority to nested fields and flattening them after the other properties.
 * @param obj - The input object.
 * @returns An object containing all the flattened properties.
 */
function flattenObjectNestedLast(obj) {
  const flattened = {};
  const nested = {};
  for (const key in obj) {
    if (Object.hasOwn(obj, key)) {
      const value = obj[key];
      if ((0, _lodash.isPlainObject)(value)) {
        nested[key] = value;
      } else {
        flattened[key] = value;
      }
    }
  }
  return {
    ...flattened,
    ...flattenObject(nested)
  };
}