"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.enableAlertsRoute = enableAlertsRoute;
var _constants = require("../../../../../common/constants");
var _rules = require("../../../../rules");
var _errors = require("../../../../lib/errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_SERVER_LOG_NAME = 'Monitoring: Write to Kibana log';
function enableAlertsRoute(server, npRoute) {
  npRoute.router.post({
    path: '/api/monitoring/v1/alerts/enable',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES cluster client'
      }
    },
    validate: false,
    options: {
      access: 'internal'
    }
  }, async (context, request, response) => {
    try {
      const alertingContext = await context.alerting;
      const actionContext = await context.actions;
      const alerts = _rules.RulesFactory.getAll();
      if (alerts.length) {
        var _npRoute$alerting, _npRoute$alerting2;
        const {
          isSufficientlySecure,
          hasPermanentEncryptionKey
        } = (_npRoute$alerting = npRoute.alerting) !== null && _npRoute$alerting !== void 0 && _npRoute$alerting.getSecurityHealth ? await ((_npRoute$alerting2 = npRoute.alerting) === null || _npRoute$alerting2 === void 0 ? void 0 : _npRoute$alerting2.getSecurityHealth()) : {
          isSufficientlySecure: false,
          hasPermanentEncryptionKey: false
        };
        if (!isSufficientlySecure || !hasPermanentEncryptionKey) {
          server.log.info(`Skipping rule creation; Stack Monitoring rules require API keys to be enabled and an encryption key to be configured.`);
          return response.ok({
            body: {
              isSufficientlySecure,
              hasPermanentEncryptionKey
            }
          });
        }
      }
      const rulesClient = await (alertingContext === null || alertingContext === void 0 ? void 0 : alertingContext.getRulesClient());
      const actionsClient = actionContext === null || actionContext === void 0 ? void 0 : actionContext.getActionsClient();
      const types = actionContext === null || actionContext === void 0 ? void 0 : actionContext.listTypes();
      if (!rulesClient || !actionsClient || !types) {
        return response.ok({
          body: undefined
        });
      }

      // Get or create the default log action
      let serverLogAction;
      const allActions = await actionsClient.getAll();
      for (const action of allActions) {
        if (action.name === DEFAULT_SERVER_LOG_NAME) {
          serverLogAction = action;
          break;
        }
      }
      if (!serverLogAction) {
        serverLogAction = await actionsClient.create({
          action: {
            name: DEFAULT_SERVER_LOG_NAME,
            actionTypeId: _constants.ALERT_ACTION_TYPE_LOG,
            config: {},
            secrets: {}
          }
        });
      }
      const actions = [{
        id: serverLogAction.id,
        config: {}
      }];
      const createdAlerts = await Promise.all(alerts.map(alert => alert.createIfDoesNotExist(rulesClient, actionsClient, actions)));
      server.log.info(`Created ${createdAlerts.length} alerts`);
      return response.ok({
        body: {
          createdAlerts
        }
      });
    } catch (err) {
      throw (0, _errors.handleError)(err);
    }
  });
}