"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.notificationsRoutes = notificationsRoutes;
var _app = require("../../common/constants/app");
var _notifications_service = require("../models/notifications_service");
var _notifications_schema = require("./schemas/notifications_schema");
var _error_wrapper = require("../client/error_wrapper");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function notificationsRoutes({
  router,
  routeGuard,
  getEnabledFeatures
}) {
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/notifications`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canGetJobs', 'ml:canGetDataFrameAnalytics', 'ml:canGetTrainedModels']
      }
    },
    summary: 'Get notifications',
    description: 'Retrieves notifications based on provided criteria.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        query: _notifications_schema.getNotificationsQuerySchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    client,
    request,
    response,
    mlSavedObjectService
  }) => {
    try {
      const notificationsService = new _notifications_service.NotificationsService(client, mlSavedObjectService, getEnabledFeatures());
      const results = await notificationsService.searchMessages(request.query);
      return response.ok({
        body: results
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/notifications/count`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canGetJobs', 'ml:canGetDataFrameAnalytics', 'ml:canGetTrainedModels']
      }
    },
    summary: 'Get notification counts',
    description: 'Counts notifications by level.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        query: _notifications_schema.getNotificationsCountQuerySchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    client,
    mlSavedObjectService,
    request,
    response
  }) => {
    try {
      const notificationsService = new _notifications_service.NotificationsService(client, mlSavedObjectService, getEnabledFeatures());
      const results = await notificationsService.countMessages(request.query);
      return response.ok({
        body: results
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
}