"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DeploymentParamsMapper = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/** Maximum allowed number of threads per allocation */
const MAX_NUMBER_OF_THREADS = 16;
/**
 * Class responsible for mapping deployment params between API and UI
 */
class DeploymentParamsMapper {
  /**
   * Gets the min allowed number of allocations.
   * - 0 for serverless and ESS with enabled autoscaling.
   * - 1 otherwise
   * @internal
   */
  get minAllowedNumberOfAllocation() {
    return this.cloudInfo.isMlAutoscalingEnabled ? 0 : 1;
  }
  constructor(mlServerLimits, cloudInfo, nlpSettings) {
    var _this$nlpSettings, _this$nlpSettings2;
    (0, _defineProperty2.default)(this, "threadingParamsValues", void 0);
    /**
     * vCPUs level breakpoints for cloud cluster with enabled ML autoscaling.
     * TODO resolve dynamically when Control Pane exposes the vCPUs range.
     */
    (0, _defineProperty2.default)(this, "autoscalingVCPUBreakpoints", void 0);
    /**
     * vCPUs level breakpoints based on the ML server limits.
     * Either on-prem or cloud with disabled ML autoscaling.
     */
    (0, _defineProperty2.default)(this, "hardwareVCPUBreakpoints", void 0);
    /**
     * Result vCPUs level breakpoint based on the cluster env.
     */
    (0, _defineProperty2.default)(this, "vCpuBreakpoints", void 0);
    this.mlServerLimits = mlServerLimits;
    this.cloudInfo = cloudInfo;
    this.nlpSettings = nlpSettings;
    this.autoscalingVCPUBreakpoints = {
      low: {
        min: this.minAllowedNumberOfAllocation,
        max: 2,
        static: 2
      },
      medium: {
        min: 1,
        max: 32,
        static: 32
      },
      high: {
        min: 1,
        max: 99999,
        static: 128
      }
    };

    /**
     * Initial value can be different for serverless and ESS with autoscaling.
     * Also not available with 0 ML active nodes.
     */
    const maxSingleMlNodeProcessors = this.mlServerLimits.max_single_ml_node_processors;
    let maxNumberOfThreads = MAX_NUMBER_OF_THREADS;
    if ((_this$nlpSettings = this.nlpSettings) !== null && _this$nlpSettings !== void 0 && _this$nlpSettings.modelDeployment) {
      maxNumberOfThreads = Math.max(...Object.values(this.nlpSettings.modelDeployment.vCPURange).map(v => v.maxThreads));
    }
    this.threadingParamsValues = Array.from(/** To allow maximum number of threads to be 2^THREADS_MAX_EXPONENT */
    {
      length: Math.floor(Math.log2(maxNumberOfThreads)) + 1
    }, (_, i) => 2 ** i).filter(maxSingleMlNodeProcessors ? v => v <= maxSingleMlNodeProcessors : () => true);
    const mediumValue = this.mlServerLimits.total_ml_processors / 2;
    this.hardwareVCPUBreakpoints = {
      low: {
        min: this.minAllowedNumberOfAllocation,
        max: 2,
        static: 2
      },
      medium: {
        min: Math.min(3, mediumValue),
        max: mediumValue,
        static: mediumValue
      },
      high: {
        min: mediumValue + 1,
        max: this.mlServerLimits.total_ml_processors,
        static: this.mlServerLimits.total_ml_processors
      }
    };
    if ((_this$nlpSettings2 = this.nlpSettings) !== null && _this$nlpSettings2 !== void 0 && _this$nlpSettings2.modelDeployment) {
      // Apply project specific overrides
      this.vCpuBreakpoints = this.nlpSettings.modelDeployment.vCPURange;
    } else if (this.cloudInfo.isMlAutoscalingEnabled) {
      this.vCpuBreakpoints = this.autoscalingVCPUBreakpoints;
    } else {
      this.vCpuBreakpoints = this.hardwareVCPUBreakpoints;
    }
  }
  getNumberOfThreads(input) {
    var _this$nlpSettings3;
    // 1 thread for ingest-optimized deployments
    if (input.optimized === 'optimizedForIngest') {
      return 1;
    }
    if ((_this$nlpSettings3 = this.nlpSettings) !== null && _this$nlpSettings3 !== void 0 && _this$nlpSettings3.modelDeployment) {
      var _this$nlpSettings$mod;
      return (_this$nlpSettings$mod = this.nlpSettings.modelDeployment.vCPURange[input.vCPUUsage]) === null || _this$nlpSettings$mod === void 0 ? void 0 : _this$nlpSettings$mod.maxThreads;
    }

    // low → 2 threads, otherwise use the maximum
    return input.vCPUUsage === 'low' ? 2 : Math.max(...this.threadingParamsValues);
  }

  /**
   * Returns allocation values accounting for the number of threads per allocation.
   * @param params
   * @internal
   */
  getAllocationsParams(params) {
    var _this$nlpSettings4;
    const threadsPerAllocation = this.getNumberOfThreads(params);
    const {
      min: minVcpu,
      max: maxVcpu
    } = this.vCpuBreakpoints[params.vCPUUsage];

    // Maximum allocations is always the upper vCPU limit divided by threads per allocation.
    const maxAllocations = Math.floor(maxVcpu / threadsPerAllocation) || 1;

    // Minimum allocations depends on environment.
    const minAllocations = (_this$nlpSettings4 = this.nlpSettings) !== null && _this$nlpSettings4 !== void 0 && _this$nlpSettings4.modelDeployment ?
    // Serverless / project-specific range comes straight from config.
    this.nlpSettings.modelDeployment.vCPURange[params.vCPUUsage].min :
    // Otherwise derive from break-points and fall back to defaults.
    Math.floor(minVcpu / threadsPerAllocation) || (params.vCPUUsage === 'low' ? this.minAllowedNumberOfAllocation : 1);
    return {
      number_of_allocations: maxAllocations,
      min_number_of_allocations: minAllocations,
      max_number_of_allocations: maxAllocations
    };
  }

  /**
   * Gets vCPU (virtual CPU) range based on the vCPU usage level
   * @param vCPUUsage
   * @returns
   */
  getVCPURange(vCPUUsage) {
    return this.vCpuBreakpoints[vCPUUsage];
  }

  /**
   * Gets VCU (Virtual Compute Units) range based on the vCPU usage level
   * @param vCPUUsage
   * @returns
   */
  getVCURange(vCPUUsage) {
    const vCPUBreakpoints = this.vCpuBreakpoints[vCPUUsage];

    // We need only min / max / static in VCUs, ignore any other props
    const allowedKeys = ['min', 'max', 'static'];
    const result = {};
    for (const key of allowedKeys) {
      const cpuValue = vCPUBreakpoints[key];
      if (cpuValue !== undefined) {
        // As we can't retrieve Search project configuration, assume vector-optimized instances
        result[key] = Math.round(cpuValue / 0.125);
      }
    }
    return result;
  }

  /**
   * Maps UI params to the actual start deployment API request
   * @param input
   */
  mapUiToApiDeploymentParams(modelId, input) {
    var _this$nlpSettings5, _this$nlpSettings5$mo;
    const resultInput = Object.create(input);
    if (((_this$nlpSettings5 = this.nlpSettings) === null || _this$nlpSettings5 === void 0 ? void 0 : (_this$nlpSettings5$mo = _this$nlpSettings5.modelDeployment) === null || _this$nlpSettings5$mo === void 0 ? void 0 : _this$nlpSettings5$mo.allowStaticAllocations) === false) {
      // Enforce adaptive resources for serverless projects with prohibited static allocations
      resultInput.adaptiveResources = true;
    }
    const allocationParams = this.getAllocationsParams(resultInput);
    return {
      modelId,
      deploymentParams: {
        deployment_id: resultInput.deploymentId,
        threads_per_allocation: this.getNumberOfThreads(resultInput),
        priority: 'normal',
        ...(!resultInput.adaptiveResources && {
          number_of_allocations: allocationParams.number_of_allocations
        })
      },
      ...(resultInput.adaptiveResources && {
        adaptiveAllocationsParams: {
          enabled: true,
          min_number_of_allocations: allocationParams.min_number_of_allocations,
          max_number_of_allocations: allocationParams.max_number_of_allocations
        }
      })
    };
  }

  /**
   * Maps deployment params from API to the UI
   * @param input
   */
  mapApiToUiDeploymentParams(input) {
    var _input$adaptive_alloc, _input$threads_per_al, _input$number_of_allo;
    let optimized = 'optimizedForIngest';
    if (input.threads_per_allocation && input.threads_per_allocation > 1) {
      optimized = 'optimizedForSearch';
    }
    const adaptiveResources = !!((_input$adaptive_alloc = input.adaptive_allocations) !== null && _input$adaptive_alloc !== void 0 && _input$adaptive_alloc.enabled);
    const vCPUs = ((_input$threads_per_al = input.threads_per_allocation) !== null && _input$threads_per_al !== void 0 ? _input$threads_per_al : 0) * (adaptiveResources ? input.adaptive_allocations.max_number_of_allocations : (_input$number_of_allo = input.number_of_allocations) !== null && _input$number_of_allo !== void 0 ? _input$number_of_allo : 0);

    // The deployment can be created via API with a number of allocations that do not exactly match our vCPU ranges.
    // In this case, we should find the closest vCPU range that does not exceed the max or static value of the range.
    const [vCPUUsage] = Object.entries(this.vCpuBreakpoints).filter(([, range]) => vCPUs <= (adaptiveResources ? range.max : range.static)).reduce((prev, curr) => {
      const prevValue = adaptiveResources ? prev[1].max : prev[1].static;
      const currValue = adaptiveResources ? curr[1].max : curr[1].static;
      return Math.abs(vCPUs - prevValue) <= Math.abs(vCPUs - currValue) ? prev : curr;
    },
    // in case allocation params exceed the max value of the high range
    ['high', this.vCpuBreakpoints.high]);
    return {
      deploymentId: input.deployment_id,
      optimized,
      adaptiveResources,
      vCPUUsage: vCPUUsage
    };
  }
}
exports.DeploymentParamsMapper = DeploymentParamsMapper;