"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useDecisionPathData = exports.processMultiClassClassificationDecisionPathData = exports.processBinaryClassificationDecisionPathData = exports.isDecisionPathData = exports.getStringBasedClassName = exports.formatValue = exports.computeMultiClassImportanceDenominator = exports.buildRegressionDecisionPathData = exports.buildClassificationDecisionPathData = exports.addAdjustedProbability = void 0;
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _mlDataFrameAnalyticsUtils = require("@kbn/ml-data-frame-analytics-utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const FEATURE_NAME = 'feature_name';
const FEATURE_IMPORTANCE = 'importance';
const RESIDUAL_IMPORTANCE_ERROR_MARGIN = 1e-5;
const decisionPathFeatureOtherTitle = _i18n.i18n.translate('xpack.ml.dataframe.analytics.decisionPathFeatureOtherTitle', {
  defaultMessage: 'other'
});
const isDecisionPathData = decisionPathData => {
  return Array.isArray(decisionPathData) && decisionPathData.length > 0 && decisionPathData[0].length === 3;
};

// cast to 'True' | 'False' | value to match Eui display
exports.isDecisionPathData = isDecisionPathData;
const getStringBasedClassName = v => {
  if (v === undefined || v === null) {
    return '';
  }
  if (typeof v === 'boolean') {
    return v ? 'True' : 'False';
  }
  if (v === 'true') return 'True';
  if (v === 'false') return 'False';
  if (typeof v === 'number') {
    return v.toString();
  }
  return v;
};
exports.getStringBasedClassName = getStringBasedClassName;
const formatValue = (number, precision = 3, fractionDigits = 1) => {
  if (Math.abs(number) < 10) {
    return Number(number.toPrecision(precision)).toString();
  }
  return number.toFixed(fractionDigits);
};
exports.formatValue = formatValue;
const useDecisionPathData = ({
  baseline,
  featureImportance,
  predictedValue,
  predictedProbability
}) => {
  const decisionPathData = (0, _react.useMemo)(() => {
    if (baseline === undefined) return;
    if ((0, _mlDataFrameAnalyticsUtils.isRegressionFeatureImportanceBaseline)(baseline)) {
      return buildRegressionDecisionPathData({
        baseline: baseline.baseline,
        featureImportance,
        predictedValue: predictedValue
      });
    }
    if ((0, _mlDataFrameAnalyticsUtils.isClassificationFeatureImportanceBaseline)(baseline)) {
      return buildClassificationDecisionPathData({
        baselines: baseline.classes,
        featureImportance,
        currentClass: predictedValue,
        predictedProbability
      });
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [baseline, featureImportance, predictedValue]);
  return {
    decisionPathData
  };
};

/**
 * Returns values to build decision path for regression jobs
 * where first data point of array is the final predicted value (end of decision path)
 */
exports.useDecisionPathData = useDecisionPathData;
const buildRegressionDecisionPathData = ({
  baseline,
  featureImportance,
  predictedValue
}) => {
  const mappedFeatureImportance = featureImportance.map(d => ({
    ...d,
    absImportance: Math.abs(d[FEATURE_IMPORTANCE])
  }));
  if (baseline && predictedValue !== undefined && Number.isFinite(predictedValue)) {
    // get the adjusted importance needed for when # of fields included in c++ analysis != max allowed
    // if num fields included = num features allowed exactly, adjustedImportance should be 0
    const adjustedImportance = predictedValue - mappedFeatureImportance.reduce((accumulator, currentValue) => accumulator + currentValue.importance, 0) - baseline;
    mappedFeatureImportance.push({
      [FEATURE_NAME]: _i18n.i18n.translate('xpack.ml.dataframe.analytics.decisionPathFeatureBaselineTitle', {
        defaultMessage: 'baseline'
      }),
      [FEATURE_IMPORTANCE]: baseline,
      absImportance: -1
    });

    // if the difference is small enough then no need to plot the residual feature importance
    if (Math.abs(adjustedImportance) > RESIDUAL_IMPORTANCE_ERROR_MARGIN) {
      mappedFeatureImportance.push({
        [FEATURE_NAME]: decisionPathFeatureOtherTitle,
        [FEATURE_IMPORTANCE]: adjustedImportance,
        absImportance: 0 // arbitrary importance so this will be of higher importance than baseline
      });
    }
  }
  const filteredFeatureImportance = mappedFeatureImportance.filter(f => f !== undefined);
  const finalResult = filteredFeatureImportance
  // sort by absolute importance so it goes from bottom (baseline) to top
  .sort((a, b) => b.absImportance - a.absImportance).map(d => [d[FEATURE_NAME], d[FEATURE_IMPORTANCE], NaN]);

  // start at the baseline and end at predicted value
  // for regression, cumulativeSum should add up to baseline
  let cumulativeSum = 0;
  for (let i = filteredFeatureImportance.length - 1; i >= 0; i--) {
    cumulativeSum += finalResult[i][1];
    finalResult[i][2] = cumulativeSum;
  }
  return finalResult;
};
exports.buildRegressionDecisionPathData = buildRegressionDecisionPathData;
const addAdjustedProbability = ({
  predictedProbability,
  decisionPlotData
}) => {
  if (predictedProbability && decisionPlotData.length > 0) {
    const adjustedResidualImportance = predictedProbability - decisionPlotData[0][2];
    // in the case where the final prediction_probability is less than the actual predicted probability
    // which happens when number of features > top_num
    // adjust the path to account for the residual feature importance as well
    if (Math.abs(adjustedResidualImportance) > RESIDUAL_IMPORTANCE_ERROR_MARGIN) {
      decisionPlotData.forEach(row => row[2] = row[2] + adjustedResidualImportance);
      decisionPlotData.push([decisionPathFeatureOtherTitle, adjustedResidualImportance, decisionPlotData[decisionPlotData.length - 1][2] - adjustedResidualImportance]);
    }
  }
  return decisionPlotData;
};
exports.addAdjustedProbability = addAdjustedProbability;
const processBinaryClassificationDecisionPathData = ({
  decisionPlotData,
  startingBaseline,
  predictedProbability
}) => {
  // array is arranged from final prediction at the top to the starting point at the bottom
  const finalResult = decisionPlotData;

  // transform the numbers into the probability space
  // starting with the baseline retrieved from trained_models metadata
  let logOddSoFar = startingBaseline;
  for (let i = finalResult.length - 1; i >= 0; i--) {
    logOddSoFar += finalResult[i][1];
    const predictionProbabilitySoFar = Math.exp(logOddSoFar) / (Math.exp(logOddSoFar) + 1);
    finalResult[i][2] = predictionProbabilitySoFar;
  }
  return addAdjustedProbability({
    predictedProbability,
    decisionPlotData: finalResult
  });
};
exports.processBinaryClassificationDecisionPathData = processBinaryClassificationDecisionPathData;
const processMultiClassClassificationDecisionPathData = ({
  baselines,
  decisionPlotData,
  startingBaseline,
  featureImportance,
  predictedProbability
}) => {
  const denominator = computeMultiClassImportanceDenominator({
    baselines,
    featureImportance
  });

  // calculate the probability path
  // p_j = exp(baseline(A) + \sum_{i=0}^j feature_importance_i(A)) / denominator
  const baseline = startingBaseline;
  let featureImportanceRunningSum = 0;
  for (let i = decisionPlotData.length - 1; i >= 0; i--) {
    featureImportanceRunningSum += decisionPlotData[i][1];
    const numerator = Math.exp(baseline + featureImportanceRunningSum);
    decisionPlotData[i][2] = numerator / denominator;
  }
  return addAdjustedProbability({
    predictedProbability,
    decisionPlotData
  });
};

/**
 * Compute the denominator used for multiclass classification
 * (\sum_{x\in{A,B,C}} exp(baseline(x) + \sum_{i=0}^j feature_importance_i(x)))
 */
exports.processMultiClassClassificationDecisionPathData = processMultiClassClassificationDecisionPathData;
const computeMultiClassImportanceDenominator = ({
  baselines,
  featureImportance
}) => {
  let denominator = 0;
  for (let x = 0; x < baselines.length; x++) {
    let featureImportanceOfClassX = 0;
    for (let i = 0; i < featureImportance.length; i++) {
      const feature = featureImportance[i];
      const classFeatureImportance = Array.isArray(feature.classes) ? feature.classes.find(c => getStringBasedClassName(c.class_name) === getStringBasedClassName(baselines[x].class_name)) : feature;
      if (classFeatureImportance && classFeatureImportance.importance !== undefined && typeof classFeatureImportance[FEATURE_IMPORTANCE] === 'number') {
        featureImportanceOfClassX += classFeatureImportance.importance;
      }
    }
    denominator += Math.exp(baselines[x].baseline + featureImportanceOfClassX);
  }
  return denominator;
};

/**
 * Returns values to build decision path for classification jobs
 * where first data point of array is the final predicted probability (end of decision path)
 */
exports.computeMultiClassImportanceDenominator = computeMultiClassImportanceDenominator;
const buildClassificationDecisionPathData = ({
  baselines,
  featureImportance,
  currentClass,
  predictedProbability
}) => {
  if (currentClass === undefined || !(Array.isArray(baselines) && baselines.length >= 2)) return [];
  const mappedFeatureImportance = featureImportance.map(feature => {
    const classFeatureImportance = Array.isArray(feature.classes) ? feature.classes.find(c => getStringBasedClassName(c.class_name) === getStringBasedClassName(currentClass)) : feature;
    if (classFeatureImportance && typeof classFeatureImportance[FEATURE_IMPORTANCE] === 'number') {
      return {
        [FEATURE_NAME]: feature[FEATURE_NAME],
        [FEATURE_IMPORTANCE]: classFeatureImportance[FEATURE_IMPORTANCE],
        absImportance: Math.abs(classFeatureImportance[FEATURE_IMPORTANCE])
      };
    }
    return undefined;
  });

  // get the baseline for the current class from the trained_models metadata
  const baselineClass = baselines.find(bl => getStringBasedClassName(bl.class_name) === getStringBasedClassName(currentClass));
  const startingBaseline = baselineClass !== null && baselineClass !== void 0 && baselineClass.baseline ? baselineClass === null || baselineClass === void 0 ? void 0 : baselineClass.baseline : 0;
  const filteredFeatureImportance = mappedFeatureImportance.filter(f => f !== undefined);
  const decisionPlotData = filteredFeatureImportance
  // sort by absolute importance so it goes from bottom (baseline) to top
  .sort((a, b) => b.absImportance - a.absImportance).map(d => [d[FEATURE_NAME], d[FEATURE_IMPORTANCE], NaN]);

  // if binary classification
  if (baselines.length === 2) {
    return processBinaryClassificationDecisionPathData({
      startingBaseline,
      decisionPlotData,
      predictedProbability
    });
  }
  // else if multiclass classification
  return processMultiClassClassificationDecisionPathData({
    baselines,
    decisionPlotData,
    startingBaseline,
    featureImportance,
    predictedProbability
  });
};
exports.buildClassificationDecisionPathData = buildClassificationDecisionPathData;