"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.computeChi2PValue = void 0;
var _critical_table_lookup = require("./critical_table_lookup");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Compute the p-value for how similar the datasets are.
 * Returned value ranges from 0 to 1, with 1 meaning the datasets are identical.
 *
 * @param {Histogram[]} normalizedBaselineTerms - An array of normalized baseline terms (Histogram objects).
 * @param {Histogram[]} normalizedDriftedTerms - An array of normalized drifted terms (Histogram objects).
 * @returns {number} The p-value indicating the similarity of the datasets.
 */
const computeChi2PValue = (normalizedBaselineTerms, normalizedDriftedTerms) => {
  // Get all unique keys from both arrays
  const allKeys = Array.from(new Set([...normalizedBaselineTerms.map(term => term.key.toString()), ...normalizedDriftedTerms.map(term => term.key.toString())])).slice(0, 100);

  // Calculate the chi-squared statistic and degrees of freedom
  let chiSquared = 0;
  const degreesOfFreedom = allKeys.length - 1;
  if (degreesOfFreedom === 0) return 1;
  allKeys.forEach(key => {
    var _driftedTerm$percenta, _baselineTerm$percent;
    const baselineTerm = normalizedBaselineTerms.find(term => term.key === key);
    const driftedTerm = normalizedDriftedTerms.find(term => term.key === key);
    const observed = (_driftedTerm$percenta = driftedTerm === null || driftedTerm === void 0 ? void 0 : driftedTerm.percentage) !== null && _driftedTerm$percenta !== void 0 ? _driftedTerm$percenta : 0;
    const expected = (_baselineTerm$percent = baselineTerm === null || baselineTerm === void 0 ? void 0 : baselineTerm.percentage) !== null && _baselineTerm$percent !== void 0 ? _baselineTerm$percent : 0;
    chiSquared += Math.pow(observed - expected, 2) / (expected > 0 ? expected : 1e-6); // Prevent divide by zero
  });

  // Use the criticalTableLookup function to determine the p-value
  return (0, _critical_table_lookup.criticalTableLookup)(chiSquared, degreesOfFreedom);
};
exports.computeChi2PValue = computeChi2PValue;