"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.StyleFieldsHelper = void 0;
exports.createStyleFieldsHelper = createStyleFieldsHelper;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _i18n = require("@kbn/i18n");
var _constants = require("../../../../common/constants");
var _get_vector_style_label = require("./components/get_vector_style_label");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function createStyleFieldsHelper(fields) {
  const promises = fields.map(async field => {
    return {
      label: await field.getLabel(),
      name: field.getName(),
      origin: field.getOrigin(),
      type: await field.getDataType(),
      supportsAutoDomain: field.supportsFieldMetaFromLocalData() || field.supportsFieldMetaFromEs(),
      isUnsupported: false
    };
  });
  const styleFields = await Promise.all(promises);
  return new StyleFieldsHelper(styleFields);
}
class StyleFieldsHelper {
  constructor(styleFields) {
    (0, _defineProperty2.default)(this, "_styleFields", void 0);
    (0, _defineProperty2.default)(this, "_ordinalAndCategoricalFields", void 0);
    (0, _defineProperty2.default)(this, "_numberFields", void 0);
    (0, _defineProperty2.default)(this, "_ordinalFields", void 0);
    const ordinalAndCategoricalFields = [];
    const numberFields = [];
    const ordinalFields = [];
    styleFields.forEach(styleField => {
      if (_constants.CATEGORICAL_DATA_TYPES.includes(styleField.type) || _constants.ORDINAL_DATA_TYPES.includes(styleField.type)) {
        ordinalAndCategoricalFields.push(styleField);
      }
      if (styleField.type === 'date' || styleField.type === 'number') {
        if (styleField.type === 'number') {
          numberFields.push(styleField);
        }
        if (styleField.supportsAutoDomain) {
          ordinalFields.push(styleField);
        }
      }
    });
    this._styleFields = styleFields;
    this._ordinalAndCategoricalFields = ordinalAndCategoricalFields;
    this._numberFields = numberFields;
    this._ordinalFields = ordinalFields;
  }
  hasFieldForStyle(field, styleName) {
    const fieldList = this._getFieldsForStyle(styleName);
    return fieldList.some(styleField => field.getName() === styleField.name);
  }
  _getFieldsForStyle(styleName) {
    switch (styleName) {
      case _constants.VECTOR_STYLES.ICON_ORIENTATION:
        return this._numberFields;
      case _constants.VECTOR_STYLES.FILL_COLOR:
      case _constants.VECTOR_STYLES.LINE_COLOR:
      case _constants.VECTOR_STYLES.LABEL_COLOR:
      case _constants.VECTOR_STYLES.LABEL_BORDER_COLOR:
      case _constants.VECTOR_STYLES.ICON:
        return this._ordinalAndCategoricalFields;
      case _constants.VECTOR_STYLES.LINE_WIDTH:
      case _constants.VECTOR_STYLES.LABEL_SIZE:
      case _constants.VECTOR_STYLES.ICON_SIZE:
        return this._ordinalFields;
      case _constants.VECTOR_STYLES.LABEL_TEXT:
        return this._styleFields;
      default:
        return [];
    }
  }
  getFieldsForStyle(styleProperty, isLayerSourceMvt) {
    const styleFields = this._getFieldsForStyle(styleProperty.getStyleName());
    return styleProperty.isDynamic() && !styleProperty.supportsFeatureState() ? styleFields.map(styleField => {
      // It is not possible to fallback to feature.properties for maplibre 'vector' source
      // Join fields that do not support feature-state can not be supported.
      if (isLayerSourceMvt && styleField.origin === _constants.FIELD_ORIGIN.JOIN) {
        return {
          ...styleField,
          isUnsupported: true,
          unsupportedMsg: _i18n.i18n.translate('xpack.maps.style.field.unsupportedWithVectorTileMsg', {
            defaultMessage: `''{styleLabel}'' does not support this field with vector tiles. To style ''{styleLabel}'' with this field, select 'Limit results' in 'Scaling'.`,
            values: {
              styleLabel: (0, _get_vector_style_label.getVectorStyleLabel)(styleProperty.getStyleName())
            }
          })
        };
      }
      return styleField;
    }) : styleFields;
  }
  getStyleFields() {
    return this._styleFields;
  }
}
exports.StyleFieldsHelper = StyleFieldsHelper;